package bin.mt.plugin;

import org.apache.commons.io.IOUtils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * MTL文件格式验证器
 * 移植自 LocalStringImpl.Parser (checkMode=true)
 */
public class MtlValidator {

    /**
     * 验证单个MTL文件
     *
     * @param file MTL文件
     * @throws MtlValidationException 如果格式不正确
     */
    public static void validate(File file) throws MtlValidationException, IOException {
        List<String> lines;
        try (var is = new FileInputStream(file)) {
            lines = IOUtils.readLines(is, StandardCharsets.UTF_8);
        }
        parse(lines, file.getPath());
    }

    /**
     * 批量验证MTL文件
     *
     * @param files MTL文件列表
     * @return 验证错误列表，空列表表示全部通过
     */
    public static List<String> validateAll(List<File> files) {
        List<String> errors = new ArrayList<>();
        for (File file : files) {
            try {
                validate(file);
            } catch (MtlValidationException e) {
                errors.add(e.getMessage());
            } catch (IOException e) {
                errors.add("Failed to read file: " + file.getPath() + " - " + e.getMessage());
            }
        }
        return errors;
    }

    private static void parse(List<String> lines, String filePath) throws MtlValidationException {
        Map<String, String> map = new HashMap<>();
        for (int lineCount = 1, size = lines.size(); lineCount <= size; lineCount++) {
            String line = lines.get(lineCount - 1);
            // 跳过注释或者空白行
            if (isCommentOrEmptyLine(line)) {
                continue;
            }
            // 查找key和value的分隔符
            int i = line.indexOf(": ");
            if (i == -1) {
                throw new MtlValidationException("Parse local string failed: " + line + " (in " + filePath + "  line: " + lineCount + ")");
            }
            // key中不能有空格
            String key = line.substring(0, i);
            if (key.indexOf(' ') != -1 || key.indexOf('\t') != -1) {
                throw new MtlValidationException("Key cannot contain spaces: " + line + " (in " + filePath + "  line: " + lineCount + ")");
            }
            // 检查key是否重复
            if (map.containsKey(key)) {
                throw new MtlValidationException("Duplicate key error: '" + key + "'" + " (in " + filePath + "  line: " + lineCount + ")");
            }
            // 解析value
            StringBuilder value = new StringBuilder();
            String originalValue = line.substring(i + 2);
            while (parseValue(originalValue, value, filePath, lineCount)) {
                lineCount++;
                if (lineCount > size) {
                    break;
                }
                originalValue = lines.get(lineCount - 1);
            }
            map.put(key, value.toString());
        }
    }

    private static boolean isCommentOrEmptyLine(String line) {
        int len = line.length();
        for (int i = 0; i < len; i++) {
            switch (line.charAt(i)) {
                case ' ':
                case '\t':
                    break;
                case '#':
                    return true;
                default:
                    return false;
            }
        }
        return true;
    }

    /**
     * @return 是否需要解析下一行
     */
    private static boolean parseValue(String line, StringBuilder result, String filePath, int lineCount) throws MtlValidationException {
        if (line.isEmpty())
            return false;
        int len = line.length();
        for (int i = 0; i < len; i++) {
            char c = line.charAt(i);
            if (c == '\\') {
                if (++i == len) {
                    return true;
                }
                c = line.charAt(i);
                switch (c) {
                    case 'n':
                        result.append('\n');
                        break;
                    case 'r':
                        result.append('\r');
                        break;
                    case 't':
                        result.append('\t');
                        break;
                    case '\\':
                        result.append('\\');
                        break;
                    case 'u':
                    case 'U':
                        if (i + 4 >= len) {
                            throw new MtlValidationException("Invalid Unicode escape character: \\" + line.substring(i) + " (in " + filePath + "  line: " + lineCount + ")");
                        } else {
                            String unicodeStr = line.substring(i + 1, i + 5);
                            try {
                                int unicodeVal = Integer.parseInt(unicodeStr, 16);
                                result.append((char) unicodeVal);
                                i += 4;
                            } catch (NumberFormatException e) {
                                throw new MtlValidationException("Invalid Unicode escape character: \\" + c + unicodeStr + " (in " + filePath + "  line: " + lineCount + ")");
                            }
                        }
                        break;
                    default:
                        throw new MtlValidationException("Invalid escape character: \\" + c + " (in " + filePath + "  line: " + lineCount + ")");
                }
            } else {
                result.append(c);
            }
        }
        return false;
    }

    public static class MtlValidationException extends Exception {
        public MtlValidationException(String message) {
            super(message);
        }
    }
}
