package bin.mt.plugin;

import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;

/**
 * 接口类验证器
 * 验证MTPluginConfig中配置的mainPreference和interfaces是否正确
 */
public class InterfaceValidator {

    // 支持的接口类型
    private static final String TRANSLATION_ENGINE = "bin.mt.plugin.api.translation.TranslationEngine";
    private static final String TEXT_EDITOR_FUNCTION = "bin.mt.plugin.api.editor.TextEditorFunction";
    private static final String TEXT_EDITOR_FLOATING_MENU = "bin.mt.plugin.api.editor.TextEditorFloatingMenu";
    private static final String TEXT_EDITOR_TOOL_MENU = "bin.mt.plugin.api.editor.TextEditorToolMenu";
    private static final String PLUGIN_PREFERENCE = "bin.mt.plugin.api.preference.PluginPreference";

    private static final String[] VALID_INTERFACE_TYPES = {
            TRANSLATION_ENGINE,
            TEXT_EDITOR_FUNCTION,
            TEXT_EDITOR_FLOATING_MENU,
            TEXT_EDITOR_TOOL_MENU
    };

    private final URLClassLoader classLoader;

    public InterfaceValidator(URLClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    /**
     * 验证配置
     *
     * @param mainPreference 主设置界面类名（可为null）
     * @param interfaces     接口类名数组（可为null）
     * @return 验证错误列表，空列表表示全部通过
     */
    public List<String> validate(String mainPreference, String[] interfaces) {
        List<String> errors = new ArrayList<>();

        // 验证interfaces
        if (interfaces != null && interfaces.length > 0) {
            for (int i = 0; i < interfaces.length; i++) {
                String interfaceName = interfaces[i];
                try {
                    validateInterface(interfaceName, i);
                } catch (InterfaceValidationException e) {
                    errors.add(e.getMessage());
                }
            }
        }

        // 验证mainPreference
        if (mainPreference != null && !mainPreference.isEmpty()) {
            try {
                validateMainPreference(mainPreference);
            } catch (InterfaceValidationException e) {
                errors.add(e.getMessage());
            }
        }

        return errors;
    }

    /**
     * 验证接口类
     */
    private void validateInterface(String className, int index) throws InterfaceValidationException {
        Class<?> clazz = loadAndCheckClass(className, "mtPlugin.interfaces[" + index + "]");

        // 检查是否实现了有效的接口类型
        boolean validInterface = false;
        for (String interfaceType : VALID_INTERFACE_TYPES) {
            try {
                Class<?> interfaceClass = Class.forName(interfaceType, false, classLoader);
                if (interfaceClass.isAssignableFrom(clazz)) {
                    validInterface = true;
                    break;
                }
            } catch (ClassNotFoundException e) {
                // 接口类不存在，跳过
            }
        }

        if (!validInterface) {
            throw new InterfaceValidationException(
                    "mtPlugin.interfaces[" + index + "] = \"" + className + "\"\n" +
                            "    Error: Class does not implement any valid interface type.\n" +
                            "    Must implement one of:\n" +
                            "      - TranslationEngine\n" +
                            "      - TextEditorFunction\n" +
                            "      - TextEditorFloatingMenu\n" +
                            "      - TextEditorToolMenu"
            );
        }
    }

    /**
     * 验证mainPreference类
     */
    private void validateMainPreference(String className) throws InterfaceValidationException {
        Class<?> clazz = loadAndCheckClass(className, "mtPlugin.mainPreference");

        // 检查是否实现了PluginPreference接口
        try {
            Class<?> pluginPreferenceClass = Class.forName(PLUGIN_PREFERENCE, false, classLoader);
            if (!pluginPreferenceClass.isAssignableFrom(clazz)) {
                throw new InterfaceValidationException(
                        "mtPlugin.mainPreference = \"" + className + "\"\n" +
                                "    Error: Class must implement PluginPreference interface."
                );
            }
        } catch (ClassNotFoundException e) {
            throw new InterfaceValidationException(
                    "mtPlugin.mainPreference = \"" + className + "\"\n" +
                            "    Error: PluginPreference interface not found in classpath.\n" +
                            "    Make sure plugin-api is in dependencies."
            );
        }
    }

    /**
     * 加载并检查类
     * - 类必须存在
     * - 类必须是public
     * - 类不能是abstract或interface
     * - 类必须有public无参构造函数
     */
    private Class<?> loadAndCheckClass(String className, String configField) throws InterfaceValidationException {
        Class<?> clazz;
        try {
            clazz = Class.forName(className, false, classLoader);
        } catch (ClassNotFoundException e) {
            throw new InterfaceValidationException(
                    configField + " = \"" + className + "\"\n" +
                            "    Error: Class not found."
            );
        }

        int modifiers = clazz.getModifiers();

        // 检查是否是public
        if (!Modifier.isPublic(modifiers)) {
            throw new InterfaceValidationException(
                    configField + " = \"" + className + "\"\n" +
                            "    Error: Class must be public."
            );
        }

        // 检查是否是abstract
        if (Modifier.isAbstract(modifiers)) {
            throw new InterfaceValidationException(
                    configField + " = \"" + className + "\"\n" +
                            "    Error: Class cannot be abstract."
            );
        }

        // 检查是否是interface
        if (Modifier.isInterface(modifiers)) {
            throw new InterfaceValidationException(
                    configField + " = \"" + className + "\"\n" +
                            "    Error: Must be a class, not an interface."
            );
        }

        // 检查是否有public无参构造函数
        try {
            Constructor<?> constructor = clazz.getConstructor();
            if (!Modifier.isPublic(constructor.getModifiers())) {
                throw new InterfaceValidationException(
                        configField + " = \"" + className + "\"\n" +
                                "    Error: Class must have a public no-arg constructor."
                );
            }
        } catch (NoSuchMethodException e) {
            throw new InterfaceValidationException(
                    configField + " = \"" + className + "\"\n" +
                            "    Error: Class must have a public no-arg constructor."
            );
        }

        return clazz;
    }

    public static class InterfaceValidationException extends Exception {
        public InterfaceValidationException(String message) {
            super(message);
        }
    }
}
