package bin.mt.plugin;

import com.android.build.api.variant.ApplicationAndroidComponentsExtension;
import com.android.build.api.variant.ApplicationVariant;
import com.android.build.api.variant.SourceDirectories;
import com.android.tools.smali.dexlib2.Opcodes;
import com.android.tools.smali.dexlib2.dexbacked.DexBackedDexFile;
import com.android.tools.smali.dexlib2.iface.ClassDef;
import com.android.tools.smali.dexlib2.writer.io.MemoryDataStore;
import com.android.tools.smali.dexlib2.writer.pool.DexPool;

import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.gradle.api.DefaultTask;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputDirectory;
import org.gradle.api.tasks.OutputDirectory;
import org.gradle.api.tasks.TaskAction;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;

public class MTPlugin implements Plugin<Project> {
    static final int PLUGIN_SDK_VERSION = 3;
    static final String LIB_PLUGIN_PUSHER = "bin.mt.plugin:pusher:1.0.0-alpha2";
    static final String LIB_PLUGIN_API = "bin.mt.plugin:api:3.0.0-alpha6";

    @Override
    public void apply(@Nonnull Project project) {
        // 确保这是一个Android项目
        if (!project.getPlugins().hasPlugin("com.android.application")) {
            throw new IllegalStateException("Packer plugin can only be applied to Android Application projects");
        }

//        // 确保资源混淆被禁用
//        if (!"false".equals(project.findProperty("android.enableResourceOptimizations"))) {
//            try {
//                File file = new File(project.getRootDir().getAbsoluteFile(), "gradle.properties");
//                String src = file.exists() ? FileUtils.readFileToString(file, StandardCharsets.UTF_8) : "";
//                if (!src.isEmpty() && !src.endsWith("\n")) {
//                    src += "\n";
//                }
//                src += "android.enableResourceOptimizations=false";
//                FileUtils.writeStringToFile(file, src, StandardCharsets.UTF_8);
//                throw new RuntimeException("File gradle.properties Changed!\nPlease Sync Project Again!");
//            } catch (IOException e) {
//                throw new RuntimeException(e);
//            }
//        }

        // 自动添加相关依赖
        project.getDependencies().add("debugImplementation", LIB_PLUGIN_PUSHER);
        project.getDependencies().add("compileOnly", LIB_PLUGIN_API);

        // 创建mtPlugin扩展配置
        project.getExtensions().create("mtPlugin", MTPluginConfig.class);

        // 配置任务
        var appExtension = project.getExtensions().getByType(ApplicationAndroidComponentsExtension.class);
        appExtension.onVariants(appExtension.selector().all(), variant -> {
            switch (variant.getName()) {
                case "debug" -> onDebugVariant(project, variant);
                case "release" -> onReleaseVariant(project, variant);
            }
        });
    }

    private void onDebugVariant(Project project, ApplicationVariant variant) {
        var generateDebugMTPluginConfigTask = project.getTasks().register("generateDebugMTPluginConfig", GenerateConfigFileTask.class, task -> {
            var minSdkVersion = variant.getMinSdk().getApiLevel();
            if (minSdkVersion < 24) {
                // https://android.googlesource.com/platform/tools/base/+/refs/heads/mirror-goog-studio-main/build-system/builder-model/src/main/java/com/android/builder/model/InjectedProperties.kt#63
                var deviceApiLevel = project.findProperty("android.injected.build.api");
                if (deviceApiLevel instanceof String str) {
                    deviceApiLevel = Integer.parseInt(str);
                }
                // https://android.googlesource.com/platform/tools/base/+/refs/heads/mirror-goog-studio-main/build-system/gradle-core/src/main/java/com/android/build/api/component/impl/features/DexingImpl.kt#192
                if (deviceApiLevel instanceof Integer level && level >= 24) {
                    minSdkVersion = 24;
                }
            }
            task.getMinAndroidVersion().set(minSdkVersion);
            task.getTimestamp().set(System.currentTimeMillis());
        });
        SourceDirectories.Flat resources = Objects.requireNonNull(variant.getSources().getResources());
        resources.addGeneratedSourceDirectory(generateDebugMTPluginConfigTask, GenerateConfigFileTask::getGeneratedResourcesDirectory);
    }

    /**
     * 生成MTPluginConfig.json文件，MT管理器会解析该文件从而生成mtp安装包
     */
    public static abstract class GenerateConfigFileTask extends DefaultTask {

        @Input
        public abstract Property<Integer> getMinAndroidVersion();

        @Input
        public abstract Property<Long> getTimestamp();

        @OutputDirectory
        public abstract DirectoryProperty getGeneratedResourcesDirectory();

        @TaskAction
        public void doTask() throws IOException {
            var mtpConfig = getProject().getExtensions().getByType(MTPluginConfig.class).validate();
            var json = mtpConfig.getConfigJson(getMinAndroidVersion().get());
            json.put("timestamp", getTimestamp().get());
            var file = getGeneratedResourcesDirectory().file("MTPluginConfig.json").get().getAsFile();
            FileUtils.writeStringToFile(file, json.toString(2), StandardCharsets.UTF_8);
        }

    }

    private void onReleaseVariant(Project project, ApplicationVariant variant) {
        project.afterEvaluate(p -> project.getTasks().register("packageReleaseMtp", PackMTPFileTask.class, task -> {
            var packageApkTask = project.getTasks().named("packageRelease").get();
            task.dependsOn(packageApkTask);

            task.setGroup("mt-plugin");
            task.setDescription("Generate MT Plugin installation package file");

            task.getMinAndroidVersion()
                    .set(variant.getMinSdk().getApiLevel());
            // 这里必须使用文件夹作为输入，不然可能出现apk文件变了，output-metadata.json文件没变的情况
            task.getInputDirectory()
                    .set(packageApkTask.getOutputs().getFiles()
                            .filter(file -> file.getName().equals("output-metadata.json"))
                            .getSingleFile()
                            .getParentFile()
                    );
            task.getOutputDirectory()
                    .set(project.getLayout().getBuildDirectory().dir("outputs/mt-plugin"));
        }));
    }

    public static abstract class PackMTPFileTask extends DefaultTask {

        @Input
        public abstract Property<Integer> getMinAndroidVersion();

        @InputDirectory
        public abstract DirectoryProperty getInputDirectory();

        @OutputDirectory
        public abstract DirectoryProperty getOutputDirectory();

        @TaskAction
        public void doTask() throws IOException {
            var mtpConfig = getProject().getExtensions().getByType(MTPluginConfig.class).validate();
            var config = mtpConfig.getConfigJson(getMinAndroidVersion().get());
            var excludedFiles = new HashSet<String>();
            var excludedDirs = new ArrayList<String>();
            for (var value : config.getJSONArray("excludedFiles")) {
                var path = value.toString();
                if (path.endsWith("/")) {
                    excludedDirs.add(path);
                } else {
                    excludedFiles.add(path);
                }
            }
            var outputDir = getOutputDirectory().getAsFile().get();
            var outputFile = new File(outputDir, mtpConfig.getPluginID() + ".mtp");
            FileUtils.forceMkdir(outputDir);

            var inputDir = getInputDirectory().getAsFile().get();
            var metaFile = new File(inputDir, "output-metadata.json");
            var metaJson = new JSONObject(FileUtils.readFileToString(metaFile, StandardCharsets.UTF_8));
            var apkFilename = metaJson.getJSONArray("elements").getJSONObject(0).getString("outputFile");

            //noinspection deprecation
            try (var zipFile = new ZipFile(new File(inputDir, apkFilename));
                 var zos = new ZipArchiveOutputStream(outputFile)) {

                // 添加manifest文件
                var manifest = config.getJSONObject("manifest");
                var manifestEntry = new ZipArchiveEntry("manifest.json");
                zos.putArchiveEntry(manifestEntry);
                zos.write(manifest.toString(2).getBytes(StandardCharsets.UTF_8));
                excludedFiles.add(manifestEntry.getName()); // 后续不要复制该文件

                // 添加dex文件
                assert manifest.getBoolean("dexMode");
                List<ZipArchiveEntry> dexEntries = getDexEntries(zipFile);
                dexEntries.stream().map(ZipArchiveEntry::getName).forEach(excludedFiles::add);
                if (dexEntries.size() == 1) {
                    // 只有一个dex文件，直接复制
                    ZipArchiveEntry dexEntry = dexEntries.get(0);
                    zos.addRawArchiveEntry(dexEntry, zipFile.getRawInputStream(dexEntry));
                } else {
                    // 尝试合并dex文件
                    List<ClassDef> classes = new ArrayList<>();
                    for (var dexEntry : dexEntries) {
                        var dexFile = DexBackedDexFile.fromInputStream(null,
                                new BufferedInputStream(zipFile.getInputStream(dexEntry)));
                        classes.addAll(dexFile.getClasses());
                    }
                    Opcodes opcodes = Opcodes.forApi(getMinAndroidVersion().get());
                    DexPool dexPool = new DexPool(opcodes);
                    int currentDexIndex = 1;
                    for (int i = 0, size = classes.size(); i < size; i++) {
                        ClassDef classDef = classes.get(i);
                        dexPool.internClass(classDef);
                        if (dexPool.hasOverflowed()) {
                            // 插件的代码量正常情况一个dex足够容下，因此这段代码基本不会运行
                            // 这里偷个懒采用比较低效的写法，影响不大
                            dexPool = new DexPool(opcodes);
                            for (int j = 0; j < i; j++) {
                                dexPool.internClass(classes.get(j));
                            }
                            writeDex(zos, dexPool, currentDexIndex++);
                            dexPool = new DexPool(opcodes);
                            dexPool.internClass(classDef);
                        }
                    }
                    writeDex(zos, dexPool, currentDexIndex);
                }

                // 复制其它文件
                var entries = zipFile.getEntriesInPhysicalOrder();
                while (entries.hasMoreElements()) {
                    var entry = entries.nextElement();
                    var name = entry.getName();
                    if (name.endsWith("/") || excludedFiles.contains(name)
                            || excludedDirs.stream().anyMatch(name::startsWith)) {
                        continue;
                    }
                    if (entry.getMethod() != ZipArchiveEntry.STORED) {
                        zos.addRawArchiveEntry(entry, zipFile.getRawInputStream(entry));
                    } else {
                        zos.putArchiveEntry(new ZipArchiveEntry(entry.getName()));
                        IOUtils.copy(zipFile.getInputStream(entry), zos);
                    }
                }
            }
        }
    }

    private static void writeDex(ZipArchiveOutputStream zos, DexPool dexPool, int dexIndex) throws IOException {
        var dataStore = new MemoryDataStore();
        dexPool.writeTo(dataStore);
        zos.putArchiveEntry(new ZipArchiveEntry(getDexName(dexIndex)));
        zos.write(dataStore.getBuffer(), 0, dataStore.getSize());
    }

    private static List<ZipArchiveEntry> getDexEntries(ZipFile zipFile) throws IOException {
        var index = 0;
        var name = getDexName(++index);
        var entry = zipFile.getEntry(name);
        if (entry == null) {
            throw new IOException("Entry not found: " + name);
        }
        var list = new ArrayList<ZipArchiveEntry>();
        while (entry != null) {
            list.add(entry);
            name = getDexName(++index);
            entry = zipFile.getEntry(name);
        }
        return list;
    }

    private static String getDexName(int index) {
        if (index <= 0) {
            throw new IllegalArgumentException();
        }
        if (index == 1)
            return "classes.dex";
        return "classes" + index + ".dex";
    }

}
