package bin.mt.plugin;

import org.json.JSONObject;

import java.util.Arrays;

/**
 * MT 插件配置
 */
public class MTPluginConfig {
    /**
     * 插件安装包 mtp 文件会被打包到当前 app 模块生成的 apk 文件内，
     * 当 apk 在您的设备上运行后会自动将 mtp 文件自动推送给 MT 管理器，
     * 因此您只需在 AS 中点击右上角菜单运行当前模块，即可实现插件的快速安装。
     * <p>
     * 您可在这里设置推送给哪个MT管理器：
     * <ul>
     * <li>自动检测请填写：auto</li>
     * <li>正式版请填写：bin.mt.plus</li>
     * <li>测试版/共存请填写：bin.mt.plus.canary</li>
     * </ul>
     * 默认为自动检测
     */
    private String pushTarget = "auto";

    /**
     * 插件ID，插件的唯一标识，类似于packageName
     * <br>
     * 只能由字母、数字、下划线和点组成
     */
    private String pluginID;

    /**
     * 插件版本号，低版本插件无法覆盖安装高版本插件
     */
    private int versionCode = 1;

    /**
     * 插件版本名称
     */
    private String versionName = "null";

    /**
     * 插件名称
     */
    private String name;

    /**
     * 插件描述
     */
    private String description;

    /**
     * 主设置界面，可不填
     */
    private String mainPreference;

    /**
     * 插件对外接口，接口类型会自动判断
     */
    private String[] interfaces;

    public String getPushTarget() {
        return pushTarget;
    }

    public void setPushTarget(String pushTarget) {
        this.pushTarget = pushTarget;
    }

    public String getPluginID() {
        return pluginID;
    }

    public void setPluginID(String pluginID) {
        this.pluginID = pluginID;
    }

    public int getVersionCode() {
        return versionCode;
    }

    public void setVersionCode(int versionCode) {
        this.versionCode = versionCode;
    }

    public String getVersionName() {
        return versionName;
    }

    public void setVersionName(String versionName) {
        this.versionName = versionName;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getMainPreference() {
        return mainPreference;
    }

    public void setMainPreference(String mainPreference) {
        this.mainPreference = mainPreference;
    }

    public String[] getInterfaces() {
        return interfaces;
    }

    public void setInterfaces(String[] interfaces) {
        this.interfaces = interfaces;
    }

    public MTPluginConfig validate() {
        if (pluginID == null || pluginID.trim().isEmpty()) {
            throw new IllegalArgumentException("mtPlugin.pluginID is required and cannot be empty");
        }

        if (!pluginID.matches("^[a-zA-Z0-9_.]+$")) {
            throw new IllegalArgumentException("mtPlugin.pluginID can only contain letters, numbers, underscores and dots");
        }

        if (versionCode <= 0) {
            throw new IllegalArgumentException("mtPlugin.versionCode must be greater than 0");
        }

        if (name == null || name.trim().isEmpty()) {
            throw new IllegalArgumentException("mtPlugin.name is required and cannot be empty");
        }

        if (description == null) {
            throw new IllegalArgumentException("mtPlugin.description is required");
        }

        return this;
    }

    public JSONObject getManifestJson(int minAndroidVersion) {
        var json = new JSONObject();
        json.put("minAndroidVersion", minAndroidVersion);
        json.put("pluginSdkVersion", MTPlugin.PLUGIN_SDK_VERSION);
        json.put("pluginID", pluginID);
        json.put("versionCode", versionCode);
        json.put("versionName", versionName);
        json.put("name", name);
        json.put("description", description);
        if (mainPreference != null && !mainPreference.isEmpty()) {
            json.put("mainPreference", mainPreference);
        }
        if (interfaces != null && interfaces.length > 0) {
            json.put("interfaces", interfaces);
        }
        json.put("dexMode", true);
        return json;
    }

    public JSONObject getConfigJson(int minAndroidVersion) {
        var json = new JSONObject();
        json.put("manifest", getManifestJson(minAndroidVersion));
        json.put("pushTarget", pushTarget);
        json.put("excludedFiles", Arrays.asList(
                "META-INF/",
                "lib/",
                "MTPluginConfig.json",
                "AndroidManifest.xml",
                "resources.arsc",
                "kotlin-tooling-metadata.json"
        ));
        return json;
    }


}
