package bin.mt.plugin;

import com.android.build.api.variant.ApplicationAndroidComponentsExtension;
import com.android.build.api.variant.ApplicationVariant;
import com.android.tools.smali.dexlib2.dexbacked.DexBackedDexFile;
import com.android.tools.smali.dexlib2.writer.io.MemoryDataStore;
import com.android.tools.smali.dexlib2.writer.pool.DexPool;

import net.lingala.zip4j.ZipFile;
import net.lingala.zip4j.model.FileHeader;

import org.gradle.api.DefaultTask;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputDirectory;
import org.gradle.api.tasks.OutputDirectory;
import org.gradle.api.tasks.TaskAction;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

public class MTPlugin implements Plugin<Project> {
    static final int PLUGIN_SDK_VERSION = 3;
    static final String LIB_PLUGIN_PUSHER = "bin.mt.plugin:pusher:1.0.0-alpha2";
    static final String LIB_PLUGIN_API = "bin.mt.plugin:api:3.0.0-alpha4";

    @Override
    public void apply(@Nonnull Project project) {
        // 确保这是一个Android项目
        if (!project.getPlugins().hasPlugin("com.android.application")) {
            throw new IllegalStateException("Packer plugin can only be applied to Android Application projects");
        }
        // 自动添加相关依赖
        project.getDependencies().add("implementation", LIB_PLUGIN_PUSHER);
        project.getDependencies().add("compileOnly", LIB_PLUGIN_API);

        // 创建mtPlugin扩展配置
        project.getExtensions().create("mtPlugin", MTPluginConfig.class);

        // 配置任务
        var appExtension = project.getExtensions().getByType(ApplicationAndroidComponentsExtension.class);
        appExtension.onVariants(appExtension.selector().all(), variant -> {
            switch (variant.getName()) {
                case "debug" -> onDebugVariant(project, variant);
                case "release" -> onReleaseVariant(project, variant);
            }
        });
    }

    private void onDebugVariant(Project project, ApplicationVariant variant) {
        var generateDebugMTPluginConfigTask = project.getTasks().register("generateDebugMTPluginConfig", GenerateConfigFileTask.class, task -> {
            var minSdkVersion = variant.getMinSdk().getApiLevel();
            if (minSdkVersion < 24) {
                var deviceApiLevel = project.findProperty("android.injected.build.api");
                if (deviceApiLevel instanceof String str) {
                    deviceApiLevel = Integer.parseInt(str);
                }
                // agp/build-system/gradle-core/src/main/java/com/android/build/api/component/impl/features/DexingImpl.kt
                if (deviceApiLevel instanceof Integer level && level >= 24) {
                    minSdkVersion = 24;
                }
            }
            task.getMinAndroidVersion().set(minSdkVersion);
        });
        Objects.requireNonNull(variant.getSources().getResources()).addGeneratedSourceDirectory(generateDebugMTPluginConfigTask, GenerateConfigFileTask::getGeneratedResourcesDirectory);
    }

    /**
     * 生成MTPluginConfig.json文件，MT管理器会解析该文件从而生成mtp安装包
     */
    public static abstract class GenerateConfigFileTask extends DefaultTask {

        @Input
        public abstract Property<Integer> getMinAndroidVersion();

        @OutputDirectory
        public abstract DirectoryProperty getGeneratedResourcesDirectory();

        @TaskAction
        public void doTask() throws IOException {
            var mtpConfig = getProject().getExtensions().getByType(MTPluginConfig.class).validate();
            var json = mtpConfig.getConfigJson(getMinAndroidVersion().get(), false);

            var file = getGeneratedResourcesDirectory().file("MTPluginConfig.json").get().getAsFile();
            try (var fos = new FileOutputStream(file)) {
                fos.write(json.toString(2).getBytes(StandardCharsets.UTF_8));
            }
        }

    }

    private void onReleaseVariant(Project project, ApplicationVariant variant) {
        project.afterEvaluate(p -> project.getTasks().register("packageReleaseMtp", PackMTPFileTask.class, task -> {
            var packageApkTask = project.getTasks().named("packageRelease").get();
            task.dependsOn(packageApkTask);

            task.setGroup("mt-plugin");
            task.setDescription("Generate MT plugin installation package file");

            task.getMinAndroidVersion()
                    .set(variant.getMinSdk().getApiLevel());
            task.getInputDirectory()
                    .set(packageApkTask.getOutputs().getFiles()
                            .filter(file -> file.getName().equals("output-metadata.json"))
                            .getSingleFile()
                            .getParentFile()
                    );
            task.getOutputDirectory()
                    .set(project.getLayout().getBuildDirectory().dir("outputs/mt-plugin"));
        }));
    }

    public static abstract class PackMTPFileTask extends DefaultTask {

        @Input
        public abstract Property<Integer> getMinAndroidVersion();

        @InputDirectory
        public abstract DirectoryProperty getInputDirectory();

        @OutputDirectory
        public abstract DirectoryProperty getOutputDirectory();

        @TaskAction
        public void doTask() throws IOException {
            var mtpConfig = getProject().getExtensions().getByType(MTPluginConfig.class).validate();
            var config = mtpConfig.getConfigJson(getMinAndroidVersion().get(), true);
            var excludedFiles = new HashSet<String>();
            var excludedDirs = new ArrayList<String>();
            for (var value : config.getJSONArray("excludedFiles")) {
                var path = value.toString();
                if (path.endsWith("/")) {
                    excludedDirs.add(path);
                } else {
                    excludedFiles.add(path);
                }
            }
            var excludedClasses = config.getJSONArray("excludedClasses").toList()
                    .stream()
                    .map(Object::toString)
                    .collect(Collectors.toSet());
            var outputDir = getOutputDirectory().getAsFile().get();
            var outputFile = new File(outputDir, mtpConfig.getPluginID() + ".mtp");
            //noinspection ResultOfMethodCallIgnored
            outputDir.mkdirs();

            var inputDir = getInputDirectory().getAsFile().get();
            var metaFile = new File(inputDir, "output-metadata.json");
            var metaJson = new JSONObject(Files.readString(metaFile.toPath(), StandardCharsets.UTF_8));
            var apkFilename = metaJson.getJSONArray("elements").getJSONObject(0).getString("outputFile");

            try (var zipFile = new ZipFile(new File(inputDir, apkFilename));
                 var zipMaker = new ZipMaker(outputFile)) {
                var manifest = config.getJSONObject("manifest");
                zipMaker.putNextEntry("manifest.json");
                zipMaker.write(manifest.toString(2).getBytes(StandardCharsets.UTF_8));
                excludedFiles.add("manifest.json");

                if (manifest.optBoolean("dexMode")) {
                    for (var dexEntry : getDexEntries(zipFile)) {
                        excludedFiles.add(dexEntry.getFileName());
                        if (excludedClasses.isEmpty()) {
                            zipMaker.copyZipEntry(dexEntry, zipFile);
                            continue;
                        }
                        var dexFile = DexBackedDexFile.fromInputStream(null, new BufferedInputStream(zipFile.getInputStream(dexEntry)));
                        if (dexFile.getClasses().stream().anyMatch(classDef -> excludedClasses.contains(classDef.getType()))) {
                            var dexBuilder = new DexPool(dexFile.getOpcodes());
                            for (var classDef : dexFile.getClasses()) {
                                if (!excludedClasses.contains(classDef.getType())) {
                                    dexBuilder.internClass(classDef);
                                }
                            }
                            var dataStore = new MemoryDataStore();
                            dexBuilder.writeTo(dataStore);
                            zipMaker.putNextEntry(dexEntry.getFileName());
                            zipMaker.write(dataStore.getBuffer(), 0, dataStore.getSize());
                        } else {
                            zipMaker.copyZipEntry(dexEntry, zipFile);
                        }
                    }
                }

                label:
                for (var entry : zipFile.getFileHeaders()) {
                    var name = entry.getFileName();
                    if (name.endsWith("/") || excludedFiles.contains(name)) {
                        continue;
                    }
                    for (int i = 0, size = excludedDirs.size(); i < size; i++) {
                        if (name.startsWith(excludedDirs.get(i))) {
                            continue label;
                        }
                    }
                    zipMaker.copyZipEntry(entry, zipFile);
                }
            }
        }
    }

    private static List<FileHeader> getDexEntries(ZipFile zipFile) throws IOException {
        var index = 0;
        var name = getDexName(++index);
        var entry = zipFile.getFileHeader(name);
        if (entry == null) {
            throw new IOException("Entry not found: " + name);
        }
        var list = new ArrayList<FileHeader>();
        while (entry != null) {
            list.add(entry);
            name = getDexName(++index);
            entry = zipFile.getFileHeader(name);
        }
        return list;
    }

    private static String getDexName(int index) {
        if (index <= 0) {
            throw new IllegalArgumentException();
        }
        if (index == 1)
            return "classes.dex";
        return "classes" + index + ".dex";
    }

}
