package bin.mt.plugin.api.util;

import java.util.concurrent.atomic.AtomicReference;

/**
 * 线程安全的UI更新器
 * <p>
 * 该类用于处理跨线程的UI更新操作，确保UI更新始终在主线程中执行。
 * 主要特性：
 * - 在UI线程中立即执行更新操作
 * - 在非UI线程中缓存数据，延迟到UI线程中处理
 * - 通过原子引用保证线程安全
 * - 避免重复缓存，只保留最新的待更新数据
 *
 * @param <T> 更新数据的类型
 */
public class UIUpdater<T> {
    /**
     * 表示未设置状态的标记对象，用于区分null值和未设置状态
     */
    private static final Object NOT_SET = new Object();

    /**
     * 原子引用，用于线程安全地存储待处理的数据
     */
    private final AtomicReference<Object> pendingData = new AtomicReference<>(NOT_SET);

    /**
     * 所属的更新器组，用于统一调度待处理的更新
     */
    private final UIUpdaterGroup updaterGroup;

    /**
     * UI更新的消费者函数，实际执行UI更新操作
     */
    private final Consumer<T> uiConsumer;

    UIUpdater(UIUpdaterGroup updaterGroup, Consumer<T> consumer) {
        this.updaterGroup = updaterGroup;
        this.uiConsumer = consumer;
    }

    /**
     * 提交数据更新请求
     * <p>
     * 根据当前线程类型采用不同的处理策略：
     * - UI线程：清除缓存数据并立即执行更新操作
     * - 非UI线程：将数据缓存到原子引用中，通过更新器组调度到UI线程处理
     *
     * @param data 要更新的数据，可以为null
     */
    public void submitUpdate(T data) {
        if (ThreadUtil.isInUiThread()) {
            pendingData.set(NOT_SET);
            uiConsumer.accept(data);
        } else {
            pendingData.set(data);
            updaterGroup.schedulePendingUpdates();
        }
    }

    /**
     * 处理待处理的更新（内部方法）
     * <p>
     * 该方法由UIUpdaterGroup在UI线程中调用，用于处理之前缓存的更新数据。
     * 使用getAndSet原子操作确保：
     * 1. 获取当前缓存的数据
     * 2. 同时将缓存重置为NOT_SET状态
     * 3. 避免重复处理同一数据
     */
    void processPendingUpdate() {
        Object data = pendingData.getAndSet(NOT_SET);
        if (data != NOT_SET) {
            //noinspection unchecked
            uiConsumer.accept((T) data);
        }
    }
}
