package bin.mt.plugin.api.util;

import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;

/**
 * 线程工具类
 * 提供UI线程相关的操作方法，包括线程判断、任务调度等功能
 */
public class ThreadUtil {

    /**
     * 主线程Handler，用于向UI线程发送消息和执行任务
     * 基于主线程Looper创建，确保所有操作都在UI线程中执行
     */
    public static final Handler HANDLER = new Handler(Looper.getMainLooper());

    /**
     * UI线程引用
     * 缓存主线程对象，用于快速判断当前线程是否为UI线程
     */
    private static final Thread UI_THREAD = Looper.getMainLooper().getThread();

    /**
     * 判断当前线程是否为UI线程
     *
     * @return true表示当前线程是UI线程，false表示不是UI线程
     */
    public static boolean isInUiThread() {
        return UI_THREAD == Thread.currentThread();
    }

    /**
     * 断言当前线程必须是UI线程
     * 如果当前线程不是UI线程，则抛出IllegalStateException异常
     *
     * @throws IllegalStateException 当前线程不是UI线程时抛出此异常
     */
    public static void assertInUIThread() {
        if (!isInUiThread()) {
            throw new IllegalStateException("This method must be called on the UI thread");
        }
    }

    /**
     * 在UI线程中执行任务
     * 如果当前已经在UI线程中，则直接执行；否则通过Handler切换到UI线程执行
     *
     * @param runnable 要执行的任务，不能为null
     */
    public static void runOnUiThread(@NonNull Runnable runnable) {
        if (isInUiThread()) {
            // 当前已在UI线程，直接执行
            runnable.run();
        } else {
            // 当前不在UI线程，通过Handler切换到UI线程执行
            HANDLER.post(runnable);
        }
    }

    /**
     * 将任务投递到UI线程执行
     * 无论当前线程是什么，都会通过Handler将任务投递到UI线程的消息队列中
     *
     * @param runnable 要执行的任务，不能为null
     */
    public static void post(@NonNull Runnable runnable) {
        HANDLER.post(runnable);
    }

    /**
     * 延迟将任务投递到UI线程执行
     * 在指定的延迟时间后，将任务投递到UI线程的消息队列中执行
     *
     * @param runnable    要执行的任务，不能为null
     * @param delayMillis 延迟时间，单位为毫秒，必须大于等于0
     */
    public static void postDelayed(@NonNull Runnable runnable, long delayMillis) {
        HANDLER.postDelayed(runnable, delayMillis);
    }
}
