package bin.mt.plugin.api.ui.dialog;

import android.text.TextUtils;

import bin.mt.plugin.api.ui.PluginProgressBar;
import bin.mt.plugin.api.ui.PluginTextView;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;
import bin.mt.plugin.api.util.ThreadUtil;
import bin.mt.plugin.api.util.UIUpdater;
import bin.mt.plugin.api.util.UIUpdaterGroup;

/**
 * 双进度对话框类，用于显示带有两个进度条的对话框
 */
public class DualProgressDialog extends BaseCancelableDialog<DualProgressDialog> {
    private final PluginDialog dialog;
    private final PluginTextView messageView;
    private final PluginProgressBar subProgressBar;
    private final PluginProgressBar totalProgressBar;
    private final UIUpdater<CharSequence> titleUpdater;
    private final UIUpdater<CharSequence> messageUpdater;
    private final UIUpdater<Integer> subProgressUpdater;
    private final UIUpdater<Integer> totalProgressUpdater;

    /**
     * 构造函数，创建进度对话框
     *
     * @param pluginUI 插件UI实例，用于构建对话框界面
     */
    public DualProgressDialog(PluginUI pluginUI) {
        super(pluginUI);
        // 构建垂直布局的对话框视图
        PluginView pluginView = pluginUI.buildVerticalLayout()
                .paddingHorizontal(pluginUI.dialogPaddingHorizontal())
                .paddingBottom(pluginUI.dialogPaddingVertical())
                .addTextView("message").singleLine().ellipsize(TextUtils.TruncateAt.END).marginTopDp(8).gone()
                .addProgressBar("subProgressBar").marginTopDp(12)
                .addTextView().text("{dual_progress_dialog_total}").marginTopDp(6)
                .addProgressBar("totalProgressBar").marginTopDp(6)
                .build();

        // 创建不可取消的对话框
        dialog = pluginUI.buildDialog().setView(pluginView).setCancelable(false).create();

        // 获取视图组件引用
        messageView = pluginView.requireViewById("message");
        subProgressBar = pluginView.requireViewById("subProgressBar");
        totalProgressBar = pluginView.requireViewById("totalProgressBar");

        // 创建UI更新器组，用于线程安全的UI更新
        UIUpdaterGroup updaterGroup = new UIUpdaterGroup();
        titleUpdater = updaterGroup.registerUpdater(dialog::setTitle);
        messageUpdater = updaterGroup.registerUpdater(text -> {
            if (text == null) {
                if (messageView.getVisibility() == PluginView.VISIBLE) {
                    messageView.setGone();
                    subProgressBar.setMarginTopDp(12);
                }
            } else {
                if (messageView.getVisibility() == PluginView.GONE) {
                    messageView.setVisible();
                    subProgressBar.setMarginTop(6);
                }
            }
            messageView.setText(text);
        });
        subProgressUpdater = updaterGroup.registerUpdater(subProgressBar::setProgress);
        totalProgressUpdater = updaterGroup.registerUpdater(totalProgressBar::setProgress);
    }

    /**
     * 设置标题文本
     *
     * @param title 要显示的标题文本
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog setTitle(CharSequence title) {
        titleUpdater.submitUpdate(title);
        return this;
    }

    /**
     * 设置消息文本
     *
     * @param message 要显示的消息文本，传入null时会隐藏消息视图
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog setMessage(CharSequence message) {
        messageUpdater.submitUpdate(message);
        return this;
    }

    /**
     * 设置子进度值
     *
     * @param subProgress 进度值
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog setSubProgress(int subProgress) {
        subProgressUpdater.submitUpdate(subProgress);
        return this;
    }

    /**
     * 设置总进度值
     *
     * @param totalProgress 总进度值
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog setTotalProgress(int totalProgress) {
        totalProgressUpdater.submitUpdate(totalProgress);
        return this;
    }

    /**
     * 设置子进度条的最大值
     *
     * @param maxProgress 进度条最大值
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog setMaxSubProgress(int maxProgress) {
        ThreadUtil.runOnUiThread(() -> subProgressBar.setMaxProgress(maxProgress));
        return this;
    }

    /**
     * 设置总进度条的最大值
     *
     * @param maxProgress 进度条最大值
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog setMaxTotalProgress(int maxProgress) {
        ThreadUtil.runOnUiThread(() -> totalProgressBar.setMaxProgress(maxProgress));
        return this;
    }

    /**
     * 获取消息视图
     *
     * @return 消息文本视图实例
     */
    public PluginTextView getMessageView() {
        return messageView;
    }

    /**
     * 获取子进度条视图
     *
     * @return 子进度条实例
     */
    public PluginProgressBar getSubProgressBar() {
        return subProgressBar;
    }

    /**
     * 获取总进度条视图
     *
     * @return 总进度条实例
     */
    public PluginProgressBar getTotalProgressBar() {
        return totalProgressBar;
    }

    @Override
    protected PluginDialog getDialog() {
        return dialog;
    }

    /**
     * 显示对话框
     * 注意：此方法必须在UI线程中调用
     *
     * @return 返回当前ProgressDialog实例
     */
    public DualProgressDialog show() {
        ThreadUtil.assertInUIThread(); // 确保在UI线程中执行
        dialog.show();
        return this;
    }

    /**
     * 检查对话框是否正在显示
     *
     * @return 如果对话框正在显示返回true，否则返回false
     */
    public boolean isShowing() {
        return dialog.isShowing();
    }

    /**
     * 关闭对话框
     * 此方法会自动切换到UI线程执行，确保线程安全
     */
    @Override
    public void dismiss() {
        ThreadUtil.runOnUiThread(() -> {
            if (dialog.isShowing()) {
                dialog.dismiss();
            }
        });
    }
}
