package bin.mt.plugin.api.ui.builder;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.ui.PluginButton;
import bin.mt.plugin.api.ui.PluginCheckBox;
import bin.mt.plugin.api.ui.PluginEditText;
import bin.mt.plugin.api.ui.PluginFrameLayout;
import bin.mt.plugin.api.ui.PluginLinearLayout;
import bin.mt.plugin.api.ui.PluginProgressBar;
import bin.mt.plugin.api.ui.PluginRadioButton;
import bin.mt.plugin.api.ui.PluginRadioGroup;
import bin.mt.plugin.api.ui.PluginSpinner;
import bin.mt.plugin.api.ui.PluginSwitchButton;
import bin.mt.plugin.api.ui.PluginTextView;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;

/**
 * 插件视图构建器接口
 * <p>
 * 这是插件UI框架中所有视图构建器的核心接口，提供了添加各种UI组件的方法。
 * 该接口采用建造者模式设计，支持链式调用来构建复杂的UI结构。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * PluginView view = pluginUI.buildVerticalLayout()
 *     .paddingDp(16, 16, 16, 16)
 *     .addTextView("title").text("欢迎使用").textSize(18)
 *     .addEditText("input").hint("请输入内容").maxLines(3)
 *     .addHorizontalLayout().gravity(Gravity.END).children(builder -> builder
 *         .addButton("cancel").text("取消").onClick(this::onCancel)
 *         .addButton("confirm").text("确认").onClick(this::onConfirm)
 *     )
 *     .build();
 * }</pre>
 * </p>
 *
 * @see PluginView
 * @see PluginUI
 * @see PluginBaseViewBuilder
 */
public interface PluginUIBuilder {

    /**
     * 获取绑定的PluginUI对象，每个PluginUIBuilder都与一个PluginUI实例关联。
     *
     * @return 绑定的PluginUI对象，永远不为null
     */
    @NonNull
    PluginUI getPluginUI();

    /**
     * 添加普通视图组件
     * <p>
     * 创建一个普通视图组件，可通过链式方式设置通用属性。
     * </p>
     *
     * @return PluginViewBuilder 普通视图构建器，用于进一步配置视图属性
     * @see PluginView
     * @see PluginViewBuilder
     */
    PluginViewBuilder addView();

    /**
     * 添加带ID的普通视图组件
     * <p>
     * 创建一个具有指定ID的普通视图组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 视图的ID，不能为null或空字符串
     * @return PluginViewBuilder 普通视图构建器，用于进一步配置视图属性
     * @see PluginView
     * @see PluginViewBuilder
     * @see PluginView#findViewById(String)
     */
    PluginViewBuilder addView(@NonNull String id);

    /**
     * 添加文本视图组件
     * <p>
     * 创建一个用于显示文本内容的视图组件。文本视图通常用于显示标签、
     * 说明文字等静态文本内容。
     * </p>
     *
     * @return PluginTextViewBuilder 文本视图构建器，用于进一步配置文本属性
     *
     * @see PluginTextView
     * @see PluginTextViewBuilder
     */
    PluginTextViewBuilder addTextView();

    /**
     * 添加带ID的文本视图组件
     * <p>
     * 创建一个具有指定ID的文本视图组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 文本视图的ID，不能为null或空字符串
     * @return PluginTextViewBuilder 文本视图构建器，用于进一步配置文本属性
     *
     * @see PluginTextView
     * @see PluginTextViewBuilder
     * @see PluginView#findViewById(String)
     */
    PluginTextViewBuilder addTextView(@NonNull String id);

    /**
     * 添加按钮组件
     * <p>
     * 创建一个可点击的按钮组件。按钮继承了文本视图的所有功能，通常用于触发用户操作。
     * </p>
     *
     * @return PluginButtonBuilder 按钮构建器，用于进一步配置按钮属性和事件
     *
     * @see PluginButton
     * @see PluginButtonBuilder
     */
    PluginButtonBuilder addButton();

    /**
     * 添加带ID的按钮组件
     * <p>
     * 创建一个具有指定ID的按钮组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 按钮的ID，不能为null或空字符串
     * @return PluginButtonBuilder 按钮构建器，用于进一步配置按钮属性和事件
     *
     * @see PluginButton
     * @see PluginButtonBuilder
     * @see PluginView#findViewById(String)
     */
    PluginButtonBuilder addButton(@NonNull String id);

    /**
     * 添加复选框组件
     * <p>
     * 创建一个复选框组件，用于多选场景。复选框具有选中/未选中两种状态，
     * 用户可以同时选择多个复选框。
     * </p>
     *
     * @return PluginCheckBoxBuilder 复选框构建器，用于进一步配置选中状态和事件监听
     *
     * @see PluginCheckBox
     * @see PluginCheckBoxBuilder
     */
    PluginCheckBoxBuilder addCheckBox();

    /**
     * 添加带ID的复选框组件
     * <p>
     * 创建一个具有指定ID的复选框组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 复选框的ID，不能为null或空字符串
     * @return PluginCheckBoxBuilder 复选框构建器，用于进一步配置选中状态和事件监听
     *
     * @see PluginCheckBox
     * @see PluginCheckBoxBuilder
     * @see PluginView#findViewById(String)
     */
    PluginCheckBoxBuilder addCheckBox(@NonNull String id);

    /**
     * 添加开关按钮组件
     * <p>
     * 创建一个开关按钮组件，提供类似物理开关的视觉效果。
     * 开关按钮具有开启/关闭两种状态，通常用于设置选项的启用/禁用。
     * </p>
     *
     * @return PluginSwitchButtonBuilder 开关按钮构建器，用于进一步配置开关状态和事件监听
     *
     * @see PluginSwitchButton
     * @see PluginSwitchButtonBuilder
     */
    PluginSwitchButtonBuilder addSwitchButton();

    /**
     * 添加带ID的开关按钮组件
     * <p>
     * 创建一个具有指定ID的开关按钮组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 开关按钮的ID，不能为null或空字符串
     * @return PluginSwitchButtonBuilder 开关按钮构建器，用于进一步配置开关状态和事件监听
     *
     * @see PluginSwitchButton
     * @see PluginSwitchButtonBuilder
     * @see PluginView#findViewById(String)
     */
    PluginSwitchButtonBuilder addSwitchButton(@NonNull String id);

    /**
     * 添加单选按钮组件
     * <p>
     * 创建一个单选按钮组件，用于单选场景。单选按钮通常与其他单选按钮
     * 组成一个单选组，在同一组中只能有一个按钮处于选中状态。
     * </p>
     *
     * @return PluginRadioButtonBuilder 单选按钮构建器，用于进一步配置选中状态和事件监听
     *
     * @see PluginRadioButton
     * @see PluginRadioButtonBuilder
     * @see #addRadioGroup(boolean)
     * @see #addRadioGroup(String, boolean)
     */
    PluginRadioButtonBuilder addRadioButton();

    /**
     * 添加带ID的单选按钮组件
     * <p>
     * 创建一个具有指定ID的单选按钮组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 单选按钮的ID，不能为null或空字符串
     * @return PluginRadioButtonBuilder 单选按钮构建器，用于进一步配置选中状态和事件监听
     *
     * @see PluginRadioButton
     * @see PluginRadioButtonBuilder
     * @see #addRadioGroup(boolean)
     * @see #addRadioGroup(String, boolean)
     * @see PluginView#findViewById(String)
     */
    PluginRadioButtonBuilder addRadioButton(@NonNull String id);

    /**
     * 添加单选按钮组容器
     * <p>
     * 创建一个指定方向的单选按钮组容器，其内的单选按钮只能有一个按钮处于选中状态
     * </p>
     *
     * @param isVertical true表示水平布局，false表示垂直布局
     * @return PluginRadioGroupBuilder 单选按钮组构建器，用于进一步配置布局属性和添加子View
     * @see PluginRadioGroup
     * @see PluginRadioGroupBuilder
     * @see PluginRadioButton
     */
    PluginRadioGroupBuilder addRadioGroup(boolean isVertical);

    /**
     * 添加单选按钮组容器
     * <p>
     * 创建一个具有指定ID和指定方向的单选按钮组容器，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id         垂直布局的ID，不能为null或空字符串
     * @param isVertical true表示水平布局，false表示垂直布局
     * @return PluginRadioGroupBuilder 单选按钮组构建器，用于进一步配置布局属性和添加子View
     * @see PluginRadioGroup
     * @see PluginRadioGroupBuilder
     * @see PluginRadioButton
     * @see PluginView#findViewById(String)
     */
    PluginRadioGroupBuilder addRadioGroup(@NonNull String id, boolean isVertical);

    /**
     * 添加下拉选择框组件
     * <p>
     * 创建一个下拉选择框组件，让用户从预定义的选项列表中选择单个选项。
     * 点击下拉框会显示选项列表，用户选择后列表收起并显示所选项。
     * </p>
     *
     * @return PluginSpinnerBuilder 下拉选择框构建器，用于进一步配置选项列表和选择事件
     *
     * @see PluginSpinner
     * @see PluginSpinnerBuilder
     */
    PluginSpinnerBuilder addSpinner();

    /**
     * 添加带ID的下拉选择框组件
     * <p>
     * 创建一个具有指定ID的下拉选择框组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 下拉选择框的ID，不能为null或空字符串
     * @return PluginSpinnerBuilder 下拉选择框构建器，用于进一步配置选项列表和选择事件
     *
     * @see PluginSpinner
     * @see PluginSpinnerBuilder
     * @see PluginView#findViewById(String)
     */
    PluginSpinnerBuilder addSpinner(@NonNull String id);

    /**
     * 添加进度条组件
     * <p>
     * 创建一个进度条组件，默认为水平风格，后续可通过{@link PluginProgressBarBuilder#style(PluginProgressBar.Style)}
     * 方法修改为圆形风格
     * </p>
     *
     * @return PluginProgressBarBuilder 进度条构建器，用于进一步配置选项列表和选择事件
     * @see PluginProgressBar
     * @see PluginProgressBarBuilder
     */
    PluginProgressBarBuilder addProgressBar();

    /**
     * 添加进度条组件
     * <p>
     * 创建一个具有指定ID的添加进度条组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 下拉选择框的ID，不能为null或空字符串
     * @return PluginProgressBarBuilder 进度条构建器，用于进一步配置选项列表和选择事件
     * @see PluginProgressBar
     * @see PluginProgressBarBuilder
     * @see PluginView#findViewById(String)
     */
    PluginProgressBarBuilder addProgressBar(@NonNull String id);

    /**
     * 添加文本编辑框组件
     * <p>
     * 创建一个文本编辑框组件，用于用户输入和编辑文本内容。
     * 支持单行和多行输入、自动换行、语法高亮、文本监听等功能。
     * </p>
     *
     * @return PluginEditTextBuilder 文本编辑框构建器，用于进一步配置输入属性和监听器
     *
     * @see PluginEditText
     * @see PluginEditTextBuilder
     */
    PluginEditTextBuilder addEditText();

    /**
     * 添加带ID的文本编辑框组件
     * <p>
     * 创建一个具有指定ID的文本编辑框组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 文本编辑框的ID，不能为null或空字符串
     * @return PluginEditTextBuilder 文本编辑框构建器，用于进一步配置输入属性和监听器
     *
     * @see PluginEditText
     * @see PluginEditTextBuilder
     * @see PluginView#findViewById(String)
     */
    PluginEditTextBuilder addEditText(@NonNull String id);

    /**
     * 添加Box风格文本编辑框组件
     * <p>
     * 普通风格下编辑框底部为一条横线，使用Box风格时编辑框背景为一个圆角矩形
     * </p>
     *
     * @return PluginEditTextBuilder 文本编辑框构建器，用于进一步配置输入属性和监听器
     *
     * @see #addEditText()
     * @see PluginEditText
     * @see PluginEditText#isBoxStyle()
     * @see PluginEditTextBuilder
     */
    PluginEditTextBuilder addEditBox();

    /**
     * 添加带ID的Box风格文本编辑框组件（别名方法）
     * <p>
     * 普通风格下编辑框底部为一条横线，使用Box风格时编辑框背景为一个圆角矩形
     * </p>
     *
     * @param id 文本编辑框的ID，不能为null或空字符串
     * @return PluginEditTextBuilder 文本编辑框构建器，用于进一步配置输入属性和监听器
     *
     * @see #addEditText(String)
     * @see PluginEditText
     * @see PluginEditText#isBoxStyle()
     * @see PluginEditTextBuilder
     * @see PluginView#findViewById(String)
     */
    PluginEditTextBuilder addEditBox(@NonNull String id);

    /**
     * 添加图片视图组件
     * <p>
     * 创建一个用于显示图片的视图组件。
     * </p>
     *
     * @return PluginImageViewBuilder 图片视图构建器，用于进一步配置图片属性和加载方式
     * @see PluginImageView
     * @see PluginImageViewBuilder
     */
    PluginImageViewBuilder addImageView();

    /**
     * 添加带ID的图片视图组件
     * <p>
     * 创建一个具有指定ID的图片视图组件，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 图片视图的ID，不能为null或空字符串
     * @return PluginImageViewBuilder 图片视图构建器，用于进一步配置图片属性和加载方式
     * @see PluginImageView
     * @see PluginImageViewBuilder
     * @see PluginView#findViewById(String)
     */
    PluginImageViewBuilder addImageView(@NonNull String id);

    /**
     * 添加水平线性布局容器
     * <p>
     * 创建一个水平方向的线性布局容器，子View将从左到右依次排列。
     * </p>
     *
     * @return PluginLinearLayoutBuilder 线性布局构建器，用于进一步配置布局属性和添加子View
     *
     * @see PluginLinearLayout
     * @see PluginLinearLayoutBuilder
     */
    PluginLinearLayoutBuilder addHorizontalLayout();

    /**
     * 添加带ID的水平线性布局容器
     * <p>
     * 创建一个具有指定ID的水平方向线性布局容器，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 水平布局的ID，不能为null或空字符串
     * @return PluginLinearLayoutBuilder 线性布局构建器，用于进一步配置布局属性和添加子View
     *
     * @see PluginLinearLayout
     * @see PluginLinearLayoutBuilder
     * @see PluginView#findViewById(String)
     */
    PluginLinearLayoutBuilder addHorizontalLayout(@NonNull String id);

    /**
     * 添加垂直线性布局容器
     * <p>
     * 创建一个垂直方向的线性布局容器，子View将从上到下依次排列。
     * </p>
     *
     * @return PluginLinearLayoutBuilder 线性布局构建器，用于进一步配置布局属性和添加子View
     *
     * @see PluginLinearLayout
     * @see PluginLinearLayoutBuilder
     */
    PluginLinearLayoutBuilder addVerticalLayout();

    /**
     * 添加带ID的垂直线性布局容器
     * <p>
     * 创建一个具有指定ID的垂直方向线性布局容器，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 垂直布局的ID，不能为null或空字符串
     * @return PluginLinearLayoutBuilder 线性布局构建器，用于进一步配置布局属性和添加子View
     *
     * @see PluginLinearLayout
     * @see PluginLinearLayoutBuilder
     * @see PluginView#findViewById(String)
     */
    PluginLinearLayoutBuilder addVerticalLayout(@NonNull String id);

    /**
     * 添加帧布局容器
     * <p>
     * 创建一个帧布局容器，子View叠加显示。
     * </p>
     *
     * @return PluginFrameLayoutBuilder 线性布局构建器，用于进一步配置布局属性和添加子View
     * @see PluginFrameLayout
     * @see PluginFrameLayoutBuilder
     */
    PluginFrameLayoutBuilder addFrameLayout();

    /**
     * 添加带ID的帧布局容器
     * <p>
     * 创建一个具有指定ID的帧布局容器，便于后续通过findViewById()方法查找。
     * </p>
     *
     * @param id 垂直布局的ID，不能为null或空字符串
     * @return PluginFrameLayoutBuilder 线性布局构建器，用于进一步配置布局属性和添加子View
     * @see PluginFrameLayout
     * @see PluginFrameLayoutBuilder
     * @see PluginView#findViewById(String)
     */
    PluginFrameLayoutBuilder addFrameLayout(@NonNull String id);

    /**
     * 完成所有UI组件的配置后，调用此方法来构建最终的PluginView实例。
     * <p>
     * 注意：仅顶层的PluginUIBuilder可调用该方法，并且只能调用一次，
     * 否则会抛出UnsupportedOperationException
     * <p>
     * 例如：
     * <pre>{@code
     * pluginUI.builderWithHorizontalLayout().addHorizontalLayout(subBuilder -> {
     *     subBuilder.addTextView().build(); // 抛出异常
     * }).build(); // 可以调用
     * }</pre>
     * </p>
     *
     * @return 构建完成的PluginView实例，可以直接用于显示或进一步操作
     *
     * @see PluginView
     */
    PluginView build() throws UnsupportedOperationException;

}
