package bin.mt.plugin.api.ui;

import android.content.res.ColorStateList;
import android.graphics.Typeface;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 插件文本视图接口
 * 继承自PluginView，提供文本显示和基本样式设置功能
 * 是所有文本相关组件的基础接口（除了PluginEditText）
 *
 * @see bin.mt.plugin.api.ui.builder.PluginTextViewBuilder
 */
public interface PluginTextView extends PluginView {

    /**
     * 获取文本内容
     *
     * @return 当前显示的文本内容
     */
    CharSequence getText();

    /**
     * 设置文本内容
     * <p>
     * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     *
     * @param text 要显示的文本内容
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    void setText(CharSequence text);

    /**
     * 在现有文本末尾追加文本
     *
     * @param text 要追加的文本内容
     */
    void append(CharSequence text);

    /**
     * 在现有文本末尾追加指定范围的文本
     *
     * @param text  源文本
     * @param start 源文本开始位置（包含）
     * @param end   源文本结束位置（不包含）
     */
    void append(CharSequence text, int start, int end);

    /**
     * 获取文本字体大小
     *
     * @return 字体大小，单位为sp
     */
    float getTextSize();

    /**
     * 设置文本字体大小
     *
     * @param sizeSp 字体大小，单位为sp
     */
    void setTextSize(float sizeSp);

    /**
     * 获取文本颜色状态列表
     * 返回包含不同状态下文本颜色定义的ColorStateList对象，
     * 可以通过这个对象获取各种状态（如pressed、focused、enabled等）下的颜色配置
     *
     * @return ColorStateList对象，包含不同状态下的文本颜色定义
     *
     * @see ColorStateList
     * @see #setTextColor(ColorStateList)
     * @see #getCurrentTextColor()
     */
    ColorStateList getTextColors();

    /**
     * 获取当前状态下的文本颜色
     * 根据控件当前的状态（如是否启用、是否按下、是否获得焦点等），
     * 从ColorStateList中解析出对应的颜色值
     *
     * @return 当前状态下的文本颜色值，格式为ARGB
     *
     * @see #getTextColors()
     * @see #setTextColor(int)
     * @see ColorStateList#getColorForState(int[], int)
     */
    int getCurrentTextColor();

    /**
     * 设置文本颜色，在任何状态下均为该颜色
     *
     * @param color 颜色值，格式为ARGB
     */
    void setTextColor(int color);

    /**
     * 设置启用和禁用状态下的文本颜色
     *
     * @param enabledColor  启用状态下的文本颜色，格式为ARGB
     * @param disabledColor 禁用状态下的文本颜色，格式为ARGB
     * @see #isEnabled()
     * @see #setEnabled(boolean)
     * @see #setTextColor(ColorStateList)
     * @see android.R.attr#state_enabled
     */
    void setTextColor(int enabledColor, int disabledColor);

    /**
     * 使用ColorStateList设置文本颜色
     * ColorStateList允许根据控件的不同状态（如pressed、focused、selected、enabled等）
     * 显示不同的颜色，提供更丰富的视觉反馈
     *
     * @param colors ColorStateList对象，定义了不同状态下的文本颜色
     * @see ColorStateList
     * @see #setTextColor(int, int)
     * @see android.R.attr#state_pressed
     * @see android.R.attr#state_focused
     * @see android.R.attr#state_selected
     * @see android.R.attr#state_enabled
     */
    void setTextColor(@NonNull ColorStateList colors);

    /**
     * 获取当前文本字体样式
     * 返回当前应用于文本的字体样式对象，包含字体族、样式（粗体、斜体等）等信息
     *
     * @return Typeface对象，表示当前的字体样式，可能为null（使用默认字体）
     *
     * @see Typeface
     */
    Typeface getTypeface();

    /**
     * 设置文本字体样式
     *
     * @param typeface 字体样式对象，如粗体、斜体等
     * @see Typeface
     */
    void setTypeface(Typeface typeface);

    /**
     * 获取文本在视图中的对齐方式
     * 返回当前设置的文本对齐方式，可能是水平对齐、垂直对齐或两者的组合
     *
     * @return 对齐方式，如Gravity.CENTER、Gravity.LEFT等
     *
     * @see android.view.Gravity
     */
    int getGravity();

    /**
     * 设置文本在视图中的对齐方式
     *
     * @param gravity 对齐方式，如Gravity.CENTER、Gravity.LEFT等
     * @see android.view.Gravity
     */
    void setGravity(int gravity);

    /**
     * 设置行间距
     *
     * @param add  额外增加的行间距，单位为像素
     * @param mult 行间距倍数，1.0表示默认行间距
     */
    void setLineSpacing(float add, float mult);

    /**
     * 设置文本省略方式
     *
     * @param truncate 省略位置，如TextUtils.TruncateAt.END表示在末尾省略
     */
    void setEllipsize(TextUtils.TruncateAt truncate);

    /**
     * 设置是否开启单行模式
     * <p>
     * 开启单行模式与调用 setLines(1) 的区别是换行符会被当作空格显示，例如对于字符串 "line\nline\nline"
     * <ul>
     * <li>setSingleLine(true) 显示为 "line line line"</li>
     * <li>setLines(1) 显示为 "line"</li>
     * </ul>
     *
     * @param singleLine 是否开启单行模式
     */
    void setSingleLine(boolean singleLine);

    /**
     * 设置固定行数
     *
     * @param lines 行数，PluginTextView将显示指定行数
     */
    void setLines(int lines);

    /**
     * 获取最小行数
     *
     * @return 最小行数
     */
    int getMinLines();

    /**
     * 设置最小行数
     *
     * @param minLines 最小行数，PluginTextView至少显示指定行数
     */
    void setMinLines(int minLines);

    /**
     * 获取最大行数
     *
     * @return 最大行数
     */
    int getMaxLines();

    /**
     * 设置最大行数
     *
     * @param maxLines 最大行数，PluginTextView最多显示指定行数
     */
    void setMaxLines(int maxLines);

    /**
     * 获取文本在显示时是否强制大写
     *
     * @return 是否强制大写
     */
    boolean isAllCaps();

    /**
     * 设置文本在显示时是否强制大写
     *
     * @param allCaps 是否强制大写
     */
    void setAllCaps(boolean allCaps);
}
