package bin.mt.plugin.api.ui;

import android.content.res.ColorStateList;
import android.graphics.Bitmap;
import android.graphics.ColorFilter;
import android.graphics.Matrix;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.widget.ImageView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * 插件图像视图接口
 * <p>
 * 图像视图（ImageView）是用于显示图片的UI组件，支持多种图片源（Drawable、Bitmap等），
 * 提供丰富的图片显示和处理功能，包括缩放模式、色调、滤镜、透明度等设置。
 * </p>
 *
 * @see bin.mt.plugin.api.ui.builder.PluginImageViewBuilder
 */
public interface PluginImageView extends PluginView {

    /**
     * 获取当前显示的图片Drawable对象
     *
     * @return 当前显示的图片Drawable对象，可能为null
     */
    Drawable getImageDrawable();

    /**
     * 设置要显示的图片Drawable对象
     * <p>
     * 用于显示矢量图、位图或其他类型的Drawable。
     * 传入null可以清除当前显示的图片。
     * </p>
     *
     * @param drawable 要显示的图片Drawable对象，可以为null
     */
    void setImage(@Nullable Drawable drawable);

    /**
     * 设置要显示的位图
     * <p>
     * 用于显示Bitmap类型的图片。
     * </p>
     *
     * @param bitmap 要显示的位图对象
     */
    void setImage(@NonNull Bitmap bitmap);

    /**
     * 获取当前设置的图片色调列表
     *
     * @return 当前的色调列表，可能为null
     * @see #setImageTintList(ColorStateList)
     */
    @Nullable
    ColorStateList getImageTintList();

    /**
     * 设置图片色调列表
     * <p>
     * 通过ColorStateList为图片应用不同状态下的颜色叠加效果。
     * 色调会根据设置的混合模式与图片原色混合，实现图片颜色的动态变化。
     * 传入null可以移除色调效果。
     * </p>
     *
     * @param tint 色调列表，可以为null
     * @see #setImageTintMode(PorterDuff.Mode)
     */
    void setImageTintList(@Nullable ColorStateList tint);

    /**
     * 获取当前设置的图片色调混合模式
     *
     * @return 当前的色调混合模式，可能为null
     * @see #setImageTintMode(PorterDuff.Mode)
     */
    @Nullable
    PorterDuff.Mode getImageTintMode();

    /**
     * 设置图片色调的混合模式
     * <p>
     * 指定色调与图片原色的混合方式，不同的模式会产生不同的视觉效果。
     * 常用模式包括SRC_IN（仅保留色调颜色）、SRC_ATOP（保留图片形状）等。
     * </p>
     *
     * @param tintMode 色调混合模式
     * @see PorterDuff.Mode
     * @see #setImageTintList(ColorStateList)
     */
    void setImageTintMode(PorterDuff.Mode tintMode);

    /**
     * 获取当前的图片缩放类型
     *
     * @return 当前的图片缩放类型
     * @see #setScaleType(ImageView.ScaleType)
     * @see ImageView.ScaleType
     */
    ImageView.ScaleType getScaleType();

    /**
     * 设置图片缩放类型
     * <p>
     * 控制图片如何在ImageView中缩放和定位。不同的缩放类型适用于不同的场景：
     * <ul>
     * <li>CENTER：图片居中显示，不缩放</li>
     * <li>CENTER_CROP：等比缩放，完全填充视图，可能裁剪图片</li>
     * <li>CENTER_INSIDE：等比缩放，完整显示图片，可能留白</li>
     * <li>FIT_CENTER：等比缩放居中，完整显示图片</li>
     * <li>FIT_XY：拉伸图片填充视图，可能变形</li>
     * </ul>
     * </p>
     *
     * @param scaleType 图片缩放类型
     * @see ImageView.ScaleType
     */
    void setScaleType(ImageView.ScaleType scaleType);

    /**
     * 获取图片的变换矩阵
     * <p>
     * 当缩放类型为MATRIX时，此矩阵用于控制图片的缩放、旋转、平移等变换。
     * </p>
     *
     * @return 图片变换矩阵
     * @see #setImageMatrix(Matrix)
     */
    Matrix getImageMatrix();

    /**
     * 设置图片的变换矩阵
     * <p>
     * 通过矩阵变换可以实现图片的任意缩放、旋转、平移、倾斜等效果。
     * 需要先将缩放类型设置为MATRIX此方法才会生效。
     * </p>
     *
     * @param matrix 图片变换矩阵
     * @see #setScaleType(ImageView.ScaleType)
     * @see ImageView.ScaleType#MATRIX
     */
    void setImageMatrix(Matrix matrix);

    /**
     * 获取是否启用了裁剪内边距
     * <p>
     * 当启用裁剪内边距时，图片会被裁剪到内边距范围内显示。
     * </p>
     *
     * @return true表示启用裁剪内边距，false表示未启用
     * @see #setCropToPadding()
     */
    boolean getCropToPadding();

    /**
     * 设置裁剪内边距
     * <p>
     * 启用后，图片将被裁剪到内边距范围内显示，
     * 超出内边距的部分不会显示。
     * </p>
     *
     * @param cropToPadding true表示启用裁剪内边距，false表示禁用
     * @see #getCropToPadding()
     */
    void setCropToPadding(boolean cropToPadding);

    /**
     * 获取当前的颜色滤镜
     *
     * @return 当前的颜色滤镜对象，可能为null
     * @see #setColorFilter(ColorFilter)
     */
    ColorFilter getColorFilter();

    /**
     * 设置颜色滤镜
     * <p>
     * 使用指定的颜色和混合模式为图片应用颜色滤镜效果。
     * 滤镜会影响整个图片的颜色显示。
     * </p>
     *
     * @param color 滤镜颜色值（ARGB格式）
     * @param mode  颜色混合模式
     * @see PorterDuff.Mode
     */
    void setColorFilter(int color, PorterDuff.Mode mode);

    /**
     * 设置颜色滤镜
     * <p>
     * 使用指定颜色和默认混合模式（SRC_ATOP）为图片应用颜色滤镜效果。
     * </p>
     *
     * @param color 滤镜颜色值（ARGB格式）
     * @see #setColorFilter(int, PorterDuff.Mode)
     */
    void setColorFilter(int color);

    /**
     * 设置颜色滤镜
     * <p>
     * 使用ColorFilter对象为图片应用自定义的颜色滤镜效果。
     * ColorFilter支持多种滤镜类型，如ColorMatrixColorFilter、LightingColorFilter等。
     * </p>
     *
     * @param cf 颜色滤镜对象
     * @see ColorFilter
     */
    void setColorFilter(ColorFilter cf);

    /**
     * 清除颜色滤镜
     * <p>
     * 移除之前设置的所有颜色滤镜效果，恢复图片原始颜色。
     * </p>
     *
     * @see #setColorFilter(int)
     * @see #setColorFilter(ColorFilter)
     */
    void clearColorFilter();

    /**
     * 获取图片的透明度
     * <p>
     * 返回图片的透明度值，范围为0（完全透明）到255（完全不透明）。
     * 注意：这与View的alpha属性不同，仅影响图片本身。
     * </p>
     *
     * @return 图片透明度值（0-255）
     * @see #setImageAlpha(int)
     */
    int getImageAlpha();

    /**
     * 设置图片的透明度
     * <p>
     * 设置图片的透明度，范围为0（完全透明）到255（完全不透明）。
     * 此方法仅影响图片的透明度，不影响ImageView本身及其背景。
     * </p>
     *
     * @param alpha 图片透明度值（0-255）
     * @see #getImageAlpha()
     */
    void setImageAlpha(int alpha);

}
