package bin.mt.plugin.api.ui;

import android.text.Editable;

import bin.mt.plugin.api.ui.builder.PluginEditTextBuilder;

/**
 * 插件文本编辑框监听器接口
 * <p>
 * 该接口类似于 {@link android.text.TextWatcher}，用于监听文本编辑框的内容变化。
 * 不同之处在于，每个回调方法都会额外传递触发事件的 {@link PluginEditText} 对象作为参数，
 * 这使得在监听器中可以直接访问和操作编辑框，而无需额外保存引用。
 * </p>
 * <p>
 * 如果不需要监听所有回调方法，可以继承 {@link Simple} 抽象类，
 * 该类提供了所有方法的空实现，只需重写感兴趣的方法即可。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * pluginUIBuilder
 *     .addEditText("input")
 *     .addEditTextChangedListener(new PluginEditTextWatcher.Simple() {
 *         @Override
 *         public void afterTextChanged(PluginEditText editText, Editable s) {
 *             // 只需重写感兴趣的方法
 *             int length = editText.length();
 *             if (length > 100) {
 *                 // 处理超长文本
 *             }
 *         }
 *     });
 * }</pre>
 * </p>
 *
 * @see PluginEditText#addTextChangedListener(PluginEditTextWatcher)
 * @see PluginEditTextBuilder#addTextChangedListener(PluginEditTextWatcher)
 * @see Simple
 * @see android.text.TextWatcher
 */
public interface PluginEditTextWatcher {

    /**
     * 文本变化前调用
     * <p>
     * 在文本内容即将发生变化时调用，此时可以记录变化前的状态。
     * </p>
     *
     * @param editText 触发文本变化的编辑框
     * @param s        变化前的文本内容
     * @param start    变化起始位置
     * @param count    将要被替换的字符数量
     * @param after    变化后将要插入的字符数量
     */
    void beforeTextChanged(PluginEditText editText, CharSequence s, int start, int count, int after);

    /**
     * 文本变化时调用
     * <p>
     * 在文本内容正在发生变化时调用，此时文本内容已经改变但尚未完成。
     * </p>
     *
     * @param editText 触发文本变化的编辑框
     * @param s        变化后的文本内容
     * @param start    变化起始位置
     * @param before   变化前被替换的字符数量
     * @param count    变化后插入的字符数量
     */
    void onTextChanged(PluginEditText editText, CharSequence s, int start, int before, int count);

    /**
     * 文本变化后调用
     * <p>
     * 在文本内容发生变化后调用，此时可以对变化后的文本进行处理。
     * 注意：在此方法中修改文本内容会再次触发文本变化事件，需要避免无限循环。
     * </p>
     *
     * @param editText 触发文本变化的编辑框
     * @param s        变化后的可编辑文本对象
     */
    void afterTextChanged(PluginEditText editText, Editable s);

    /**
     * PluginEditTextWatcher 的简化抽象实现类
     */
    abstract class Simple implements PluginEditTextWatcher {

        @Override
        public void beforeTextChanged(PluginEditText editText, CharSequence s, int start, int count, int after) {

        }

        @Override
        public void onTextChanged(PluginEditText editText, CharSequence s, int start, int before, int count) {

        }

        @Override
        public void afterTextChanged(PluginEditText editText, Editable s) {

        }
    }
}
