package bin.mt.plugin.api.translation;

import androidx.annotation.NonNull;

import java.io.IOException;
import java.util.List;

import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 翻译引擎接口，用于MT管理器的翻译模式和文本输入框的翻译功能
 * <p>
 * <b>生命周期：</b>
 * <ol>
 *   <li>{@link #init(PluginContext)} - 引擎初始化，每个实例仅调用一次</li>
 *   <li>{@link #beforeStart()} - 翻译任务开始前的回调（UI线程）</li>
 *   <li>{@link #onStart()} - 翻译任务开始时的回调（子线程）</li>
 *   <li>{@link #translate(String, String, String)} - 执行翻译，根据任务需求可能调用一次或多次（子线程）</li>
 *   <li>{@link #onFinish()} - 翻译任务结束时的回调（子线程）</li>
 *   <li>{@link #afterFinish()} - 翻译任务结束后的回调（UI线程）</li>
 * </ol>
 * <p>
 * <b>注意：</b>如果在 {@link #translate} 过程中发生错误，将调用 {@link #onError(Exception)}，
 * 且不会再调用 {@link #onFinish()} 和 {@link #afterFinish()}。
 * <p>
 * <b>快速开始：</b>
 * 建议继承 {@link BaseTranslationEngine} 而非直接实现此接口，基类提供了常用的默认实现。
 *
 * @see BaseTranslationEngine
 * @see BatchTranslationEngine
 */
public interface TranslationEngine {

    /**
     * 初始化翻译引擎
     * <p>
     * 此方法在引擎实例创建后调用，且每个实例仅调用一次。
     *
     * @param context MT插件上下文，用于访问插件资源和MT功能
     */
    void init(PluginContext context);

    /**
     * 获取插件上下文
     *
     * @return {@link #init(PluginContext)} 中传入的上下文对象
     */
    PluginContext getContext();

    /**
     * 获取翻译引擎的显示名称
     * <p>
     * 该名称将显示在翻译引擎选择列表中。如果返回的名称为 <code>{key}</code> 格式，
     * 将自动转换为本地化文本，具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}。
     *
     * @return 翻译引擎名称，不能为null
     */
    @NonNull
    String name();

    /**
     * 获取翻译引擎的配置信息
     *
     * @return 配置对象，不能为null
     * @see Configuration
     * @see ConfigurationBuilder
     */
    @NonNull
    Configuration getConfiguration();

    /**
     * 加载支持的源语言列表
     * <p>
     * 返回的语言代码将用于 {@link #loadTargetLanguages(String)} 和
     * {@link #translate(String, String, String)} 方法。
     *
     * @return 源语言代码列表，不能为null
     */
    @NonNull
    List<String> loadSourceLanguages();

    /**
     * 加载支持的目标语言列表
     * <p>
     * 如果 {@link Configuration#targetLanguageMutable} 为 true，每当用户切换源语言时
     * 都会调用此方法来更新目标语言列表；为 false 时仅在界面初始化时调用一次，
     * 此时 sourceLanguage 参数没有实际意义。
     *
     * @param sourceLanguage 源语言代码，仅当 targetLanguageMutable 为 true 时有意义
     * @return 目标语言代码列表，不能为null
     */
    @NonNull
    List<String> loadTargetLanguages(String sourceLanguage);

    /**
     * 将语言代码转换为用户友好的显示名称
     * <p>
     * 用于在界面上展示 {@link #loadSourceLanguages()} 和
     * {@link #loadTargetLanguages(String)} 返回的语言代码。
     *
     * @param language 语言代码，如 "en"、"zh-CN"
     * @return 语言的显示名称，如 "English"、"简体中文"
     */
    @NonNull
    String getLanguageDisplayName(String language);

    /**
     * 翻译开始前的回调
     * <p>
     * 在 {@link #onStart()} 之前调用，运行在<b>UI线程</b>，禁止执行耗时操作或网络请求。
     */
    void beforeStart();

    /**
     * 翻译开始时的回调
     * <p>
     * 在 {@link #beforeStart()} 之后、{@link #translate} 之前调用，运行在<b>子线程</b>，可执行耗时操和或网络请求。
     */
    void onStart();

    /**
     * 执行文本翻译
     * <p>
     * 此方法运行在<b>子线程</b>，可能被<b>多次调用</b>以翻译不同的文本。
     *
     * @param text           待翻译的文本。在翻译模式功能中，如果 {@link Configuration#acceptTranslated}
     *                       为false，则始终传入原始文本；为true时可能传入已翻译的文本
     * @param sourceLanguage 源语言代码
     * @param targetLanguage 目标语言代码
     * @return 翻译后的文本，不能为null
     * @throws IOException 翻译过程中发生的IO异常（如网络错误）
     */
    @NonNull
    String translate(String text, String sourceLanguage, String targetLanguage) throws IOException;

    /**
     * 翻译结束时的回调
     * <p>
     * 在 {@link #translate} 之后、{@link #afterFinish()} 之前调用，运行在<b>子线程</b>，可执行耗时操作和网络请求。
     */
    void onFinish();

    /**
     * 翻译结束后的回调
     * <p>
     * 在 {@link #onFinish()} 之后调用，运行在<b>UI线程</b>，禁止执行耗时操作或网络请求。
     */
    void afterFinish();

    /**
     * 翻译错误处理回调
     * <p>
     * 当 {@link #translate} 过程中发生异常时调用，运行在<b>UI线程</b>，可用于自定义错误提示。
     * <p>
     * <b>注意：</b>调用此方法后，{@link #onFinish()} 和 {@link #afterFinish()} 将不会被调用。
     *
     * @param e 发生的异常
     * @return 如果返回true，MT将不显示默认的错误对话框；返回false则显示默认错误提示
     */
    boolean onError(Exception e);

    /**
     * 翻译引擎配置构建器
     * <p>
     * 使用建造者模式创建 {@link Configuration} 对象，提供链式调用支持。
     *
     * <pre>{@code
     * Configuration config = new ConfigurationBuilder()
     *     .setAcceptTranslated(false)
     *     .setAutoRepairFormatSpecifiersError(true)
     *     .build();
     * }</pre>
     */
    class ConfigurationBuilder {
        private boolean acceptTranslated;
        private boolean forceNotToSkipTranslated;
        private boolean targetLanguageMutable;
        private boolean autoRepairFormatSpecifiersError;
        private boolean disableAutoHideLanguage;

        /**
         * 在翻译模式功能中，设置是否接受已翻译过的文本作为输入
         * <p>
         * 例如词条 "Apple" 第一次翻译结果为 "苹果"，再次翻译时：
         * <ul>
         *   <li>false（默认）- 传入原始文本 "Apple"</li>
         *   <li>true - 传入已翻译的文本 "苹果"</li>
         * </ul>
         *
         * @param acceptTranslated 是否接受已翻译文本
         * @return 当前构建器实例
         */
        public ConfigurationBuilder setAcceptTranslated(boolean acceptTranslated) {
            this.acceptTranslated = acceptTranslated;
            return this;
        }

        /**
         * 在翻译模式功能中，设置是否强制不跳过已翻译的词条
         * <p>
         * 启用后，翻译模式界面上的"跳过已翻译"选项将被隐藏，所有词条都会被翻译。
         * 默认为 false。
         *
         * @param forceNotToSkipTranslated 是否强制不跳过
         * @return 当前构建器实例
         */
        public ConfigurationBuilder setForceNotToSkipTranslated(boolean forceNotToSkipTranslated) {
            this.forceNotToSkipTranslated = forceNotToSkipTranslated;
            return this;
        }

        /**
         * 设置目标语言列表是否随源语言变化
         * <p>
         * 某些翻译引擎的目标语言列表会根据所选源语言动态变化，
         * 此时应设为 true，MT会在源语言改变时重新调用
         * {@link TranslationEngine#loadTargetLanguages(String)}。
         * 默认为 false。
         *
         * @param targetLanguageMutable 目标语言是否可变
         * @return 当前构建器实例
         */
        public ConfigurationBuilder setTargetLanguageMutable(boolean targetLanguageMutable) {
            this.targetLanguageMutable = targetLanguageMutable;
            return this;
        }

        /**
         * 设置是否自动修复格式化占位符的翻译错误
         * <p>
         * 某些翻译引擎可能会错误地翻译格式化占位符，如将 "%s" 翻译为 "％S"。
         * 启用此选项后，MT会自动修复这类常见错误。
         * 默认为 false。
         *
         * @param autoRepairFormatSpecifiersError 是否自动修复
         * @return 当前构建器实例
         */
        public ConfigurationBuilder setAutoRepairFormatSpecifiersError(boolean autoRepairFormatSpecifiersError) {
            this.autoRepairFormatSpecifiersError = autoRepairFormatSpecifiersError;
            return this;
        }

        /**
         * 设置是否自动修复格式化占位符的翻译错误
         *
         * @param autoRepairFormatSpecifiersError 是否自动修复
         * @return 当前构建器实例
         * @see #setAutoRepairFormatSpecifiersError(boolean)
         * @deprecated 方法名拼写不准确，请使用 {@link #setAutoRepairFormatSpecifiersError(boolean)} 代替
         */
        @Deprecated
        public ConfigurationBuilder setAutoRepairFormatControlError(boolean autoRepairFormatSpecifiersError) {
            this.autoRepairFormatSpecifiersError = autoRepairFormatSpecifiersError;
            return this;
        }

        /**
         * 设置是否禁用语言自动隐藏功能
         * <p>
         * 对于支持大量语言的翻译引擎（如Google翻译），用户可以在翻译模式中
         * 隐藏不常用的语言。如果不希望引擎的语言被用户隐藏，可设为 true。
         * 默认为 false。
         *
         * @param disableAutoHideLanguage 是否禁用自动隐藏
         * @return 当前构建器实例
         */
        public ConfigurationBuilder setDisableAutoHideLanguage(boolean disableAutoHideLanguage) {
            this.disableAutoHideLanguage = disableAutoHideLanguage;
            return this;
        }

        /**
         * 构建配置对象
         *
         * @return 不可变的 {@link Configuration} 实例
         */
        public Configuration build() {
            return new Configuration(acceptTranslated,
                    forceNotToSkipTranslated,
                    targetLanguageMutable,
                    autoRepairFormatSpecifiersError,
                    disableAutoHideLanguage);
        }
    }

    /**
     * 翻译引擎配置类
     * <p>
     * 不可变对象，包含翻译引擎的所有配置参数。
     * 使用 {@link ConfigurationBuilder} 创建实例。
     *
     * @see ConfigurationBuilder
     */
    class Configuration {
        /**
         * 在翻译模式功能中，是否接受已翻译的文本作为翻译输入
         *
         * @see ConfigurationBuilder#setAcceptTranslated(boolean)
         */
        public final boolean acceptTranslated;

        /**
         * 在翻译模式功能中，是否强制不跳过已翻译的词条
         *
         * @see ConfigurationBuilder#setForceNotToSkipTranslated(boolean)
         */
        public final boolean forceNotToSkipTranslated;

        /**
         * 目标语言列表是否随源语言变化
         *
         * @see ConfigurationBuilder#setTargetLanguageMutable(boolean)
         */
        public final boolean targetLanguageMutable;

        /**
         * 是否自动修复格式化占位符的翻译错误
         *
         * @see ConfigurationBuilder#setAutoRepairFormatSpecifiersError(boolean)
         */
        public final boolean autoRepairFormatSpecifiersError;

        /**
         * 是否禁用语言自动隐藏功能
         *
         * @see ConfigurationBuilder#setDisableAutoHideLanguage(boolean)
         */
        public final boolean disableAutoHideLanguage;

        /**
         * 创建配置对象
         * <p>
         * 此构造方法仅供 {@link ConfigurationBuilder} 内部使用，
         * 请使用 {@link ConfigurationBuilder#build()} 创建实例。
         */
        Configuration(boolean acceptTranslated,
                      boolean forceNotToSkipTranslated,
                      boolean targetLanguageMutable,
                      boolean autoRepairFormatSpecifiersError,
                      boolean disableAutoHideLanguage) {
            this.acceptTranslated = acceptTranslated;
            this.forceNotToSkipTranslated = forceNotToSkipTranslated;
            this.targetLanguageMutable = targetLanguageMutable;
            this.autoRepairFormatSpecifiersError = autoRepairFormatSpecifiersError;
            this.disableAutoHideLanguage = disableAutoHideLanguage;
        }
    }
}
