package bin.mt.plugin.api.translation;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.PluginContext;

/**
 * 翻译引擎抽象基类，提供 {@link TranslationEngine} 接口的基础实现。
 * <p>
 * 此类为开发者提供了便捷的翻译引擎开发方式，内置了常用的默认实现：
 * <ul>
 *   <li>自动管理 {@link PluginContext} 的存储和获取</li>
 *   <li>提供简化的 {@link #init()} 方法供子类重写</li>
 *   <li>默认启用格式化占位符自动修复功能</li>
 *   <li>提供基于MT内置语言库的 {@link #getLanguageDisplayName(String)} 实现</li>
 *   <li>生命周期方法的空实现，子类可按需重写</li>
 * </ul>
 * <p>
 * <b>快速开始：</b>
 * <pre>{@code
 * public class MyTranslationEngine extends BaseTranslationEngine {
 *
 *     @NonNull
 *     @Override
 *     public String name() {
 *         return "我的翻译引擎";
 *     }
 *
 *     @NonNull
 *     @Override
 *     public List<String> loadSourceLanguages() {
 *         return Arrays.asList("en", "zh-CN");
 *     }
 *
 *     @NonNull
 *     @Override
 *     public List<String> loadTargetLanguages(String sourceLanguage) {
 *         return Arrays.asList("en", "zh-CN");
 *     }
 *
 *     @NonNull
 *     @Override
 *     public String translate(String text, String sourceLanguage, String targetLanguage) {
 *         // 调用翻译API
 *         return translatedText;
 *     }
 * }
 * }</pre>
 *
 * @see TranslationEngine
 * @see BatchTranslationEngine
 * @see BaseBatchTranslationEngine
 */
public abstract class BaseTranslationEngine implements TranslationEngine {

    /**
     * 翻译引擎配置对象
     */
    private final Configuration configuration;

    /**
     * MT插件上下文，在 {@link #init(PluginContext)} 中初始化
     */
    private PluginContext context;

    /**
     * 使用默认配置构造翻译引擎
     * <p>
     * 默认配置启用了格式化占位符自动修复功能
     * （{@link ConfigurationBuilder#setAutoRepairFormatSpecifiersError(boolean)} 为 true）。
     */
    public BaseTranslationEngine() {
        this.configuration = new ConfigurationBuilder()
                .setAutoRepairFormatSpecifiersError(true)
                .build();
    }

    /**
     * 使用自定义配置构造翻译引擎
     *
     * @param configuration 翻译引擎配置对象
     * @see ConfigurationBuilder
     */
    public BaseTranslationEngine(Configuration configuration) {
        this.configuration = configuration;
    }

    /**
     * 初始化翻译引擎
     * <p>
     * 此方法被声明为 final，子类不能重写。方法内部会保存 context 并调用
     * 可重写的 {@link #init()} 方法。子类应重写无参的 {@link #init()} 方法
     * 来执行自定义初始化逻辑。
     *
     * @param context MT插件上下文对象
     */
    @Override
    public final void init(PluginContext context) {
        this.context = context;
        init();
    }

    /**
     * 子类可重写的初始化方法
     * <p>
     * 此方法在 {@link #init(PluginContext)} 中被调用，此时 context 已设置完成，
     * 可通过 {@link #getContext()} 获取。子类可在此方法中进行具体的初始化操作
     * <p>
     * 默认实现为空。
     */
    protected void init() {
        // 默认空实现，子类可重写
    }

    /**
     * 获取MT插件上下文
     *
     * @return 在 {@link #init(PluginContext)} 中传入的上下文对象
     */
    @Override
    public PluginContext getContext() {
        return context;
    }

    /**
     * 获取翻译引擎配置
     *
     * @return 构造时传入或默认创建的配置对象
     */
    @NonNull
    @Override
    public Configuration getConfiguration() {
        return configuration;
    }

    /**
     * 获取语言的本地化显示名称
     * <p>
     * 使用MT内置的语言名称本地化功能，通过 "lang:" 前缀从上下文获取语言显示名称。
     * 如果未找到对应的本地化名称，则返回原始语言代码。
     *
     * @param language 语言代码，如 "en"、"zh-CN"、"ja"
     * @return 语言的本地化显示名称，如 "English"、"简体中文"、"日本語"
     */
    @NonNull
    @Override
    public String getLanguageDisplayName(String language) {
        String name = getContext().getStringNullable("lang:" + language);
        return name != null ? name : language;
    }

    /**
     * 翻译开始前的回调
     * <p>
     * 在 {@link #onStart()} 之前调用，运行在<b>UI线程</b>，禁止执行耗时操作或网络请求。
     * 默认实现为空，子类可重写此方法进行UI相关的准备操作。
     */
    @Override
    public void beforeStart() {
        // 默认空实现
    }

    /**
     * 翻译开始前的回调
     * <p>
     * 在 {@link #beforeStart()} 之后、{@link #translate} 之前调用，运行在<b>子线程</b>，可执行耗时操和或网络请求。
     * 默认实现为空，子类可重写此方法进行网络连接建立等操作。
     */
    @Override
    public void onStart() {
        // 默认空实现
    }

    /**
     * 翻译结束或用户取消后的回调
     * <p>
     * 在 {@link #translate} 之后、{@link #afterFinish()} 之前调用，运行在<b>子线程</b>，可执行耗时操作和网络请求。
     * 默认实现为空，子类可重写此方法进行资源释放等操作。
     */
    @Override
    public void onFinish() {
        // 默认空实现
    }

    /**
     * 翻译结束或用户取消后的回调
     * <p>
     * 在 {@link #onFinish()} 之后调用，运行在<b>UI线程</b>，禁止执行耗时操作或网络请求。
     * 默认实现为空，子类可重写此方法进行UI相关的清理操作。
     */
    @Override
    public void afterFinish() {
        // 默认空实现
    }

    /**
     * 翻译错误处理
     * <p>
     * 运行在<b>UI线程</b>，当翻译过程中发生异常时调用。
     * 默认实现将异常记录到日志并返回 false（让MT显示默认错误对话框）。
     * 子类可重写此方法实现自定义的错误处理逻辑。
     * <p>
     * <b>注意：</b>调用此方法后，{@link #onFinish()} 和 {@link #afterFinish()} 将不会被调用。
     *
     * @param e 发生的异常
     * @return true 表示已自行处理错误，MT不显示错误对话框；
     *         false 表示让MT显示默认的错误提示对话框
     */
    @Override
    public boolean onError(Exception e) {
        getContext().log(e);
        return false;
    }
}
