package bin.mt.plugin.api.translation;

import androidx.annotation.NonNull;

import java.io.IOException;

/**
 * 批量翻译引擎抽象基类，提供 {@link BatchTranslationEngine} 接口的基础实现。
 * <p>
 * 此类继承自 {@link BaseTranslationEngine}，在其基础上增加了批量翻译支持。
 * 子类只需实现 {@link #batchTranslate} 方法，{@link #translate} 方法会自动
 * 委托给批量翻译实现。
 * <p>
 * <b>快速开始：</b>
 * <pre>{@code
 * public class MyBatchTranslationEngine extends BaseBatchTranslationEngine {
 *
 *     @NonNull
 *     @Override
 *     public String name() {
 *         return "我的批量翻译引擎";
 *     }
 *
 *     @NonNull
 *     @Override
 *     public List<String> loadSourceLanguages() {
 *         return Arrays.asList("en", "zh-CN");
 *     }
 *
 *     @NonNull
 *     @Override
 *     public List<String> loadTargetLanguages(String sourceLanguage) {
 *         return Arrays.asList("en", "zh-CN");
 *     }
 *
 *     @NonNull
 *     @Override
 *     public String[] batchTranslate(String[] texts, String sourceLanguage, String targetLanguage) {
 *         // 调用支持批量翻译的API
 *         return translatedTexts;
 *     }
 * }
 * }</pre>
 *
 * @see BatchTranslationEngine
 * @see BaseTranslationEngine
 */
public abstract class BaseBatchTranslationEngine extends BaseTranslationEngine implements BatchTranslationEngine {

    /**
     * 使用默认配置构造批量翻译引擎
     * <p>
     * 默认配置启用了格式化占位符自动修复功能
     * （{@link ConfigurationBuilder#setAutoRepairFormatSpecifiersError(boolean)} 为 true）。
     */
    public BaseBatchTranslationEngine() {
    }

    /**
     * 使用自定义配置构造批量翻译引擎
     *
     * @param configuration 翻译引擎配置对象
     * @see ConfigurationBuilder
     */
    public BaseBatchTranslationEngine(Configuration configuration) {
        super(configuration);
    }

    /**
     * 创建分批策略
     * <p>
     * 默认实现基于词条数和字符数进行限制：单批次最多 50 条，总字符数最多 5000。
     * 子类可重写此方法以提供自定义的拆分策略。
     *
     * @return 分批策略实例
     */
    @Override
    public BatchingStrategy createBatchingStrategy() {
        return new DefaultBatchingStrategy(50, 5000);
    }

    /**
     * 执行单个文本翻译
     * <p>
     * 此方法将单个文本包装为数组，委托给 {@link #batchTranslate} 处理，
     * 然后返回结果数组的第一个元素。子类通常不需要重写此方法。
     *
     * @param text           待翻译的文本
     * @param sourceLanguage 源语言代码
     * @param targetLanguage 目标语言代码
     * @return 翻译后的文本
     * @throws IOException 翻译过程中发生的IO异常
     */
    @NonNull
    @Override
    public String translate(String text, String sourceLanguage, String targetLanguage) throws IOException {
        return batchTranslate(new String[]{text}, sourceLanguage, targetLanguage)[0];
    }
}
