package bin.mt.plugin.api.regex;

import java.util.regex.PatternSyntaxException;

/**
 * 正则表达式匹配结果快照
 * <p>
 * 保存某次匹配操作的不可变状态，包括匹配位置、捕获组内容等信息。
 * 快照对象独立于原Matcher对象，可以在Matcher继续执行其他匹配操作后
 * 仍然访问之前的匹配结果。
 * <p>
 * 典型使用场景：
 * - 需要保存多个匹配结果以便后续处理
 * - 在继续查找下一个匹配前保存当前匹配状态
 * - 避免重复执行耗时的匹配操作
 */
public interface MatcherSnapshot {

    /**
     * 获取快照关联的文本序列
     *
     * @return 匹配时的字符序列
     */
    CharSequence getText();

    /**
     * 获取匹配记录中匹配子序列的起始位置
     * <p>
     * 等同于start(0)
     *
     * @return 匹配子序列的起始索引
     */
    int start();

    /**
     * 获取匹配记录中指定捕获组的起始位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的起始索引，如果该组未参与匹配则返回-1
     */
    int start(int group);

    /**
     * 获取匹配记录中匹配子序列的结束位置
     * <p>
     * 等同于end(0)
     *
     * @return 匹配子序列的结束索引（不包含该位置的字符）
     */
    int end();

    /**
     * 获取匹配记录中指定捕获组的结束位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的结束索引（不包含该位置的字符），如果该组未参与匹配则返回-1
     */
    int end(int group);

    /**
     * 获取匹配记录中匹配的子序列内容
     * <p>
     * 等同于group(0)
     *
     * @return 匹配的字符串内容
     */
    String group();

    /**
     * 获取匹配记录中指定捕获组的内容
     *
     * @param group 捕获组的索引（0表示整个匹配，1及以上表示各个捕获组）
     * @return 指定捕获组的字符串内容，如果该组未参与匹配则返回null
     */
    String group(int group);

    /**
     * 获取正则表达式中捕获组的数量
     * <p>
     * 组0（整个匹配）不包含在计数中
     *
     * @return 捕获组的数量（不包括组0，即整个匹配）
     */
    int groupCount();

    /**
     * 展开替换模板字符串，具体介绍请查看 {@link  Matcher#expandReplacement(String)}
     *
     * @param replacementTemplate 替换模板字符串
     * @return 展开后的替换文本
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see Matcher#expandReplacement(String)
     */
    String expandReplacement(String replacementTemplate);

    /**
     * 将展开的替换内容追加到StringBuilder中
     * <p>
     * 功能同expandReplacement()，但直接追加到提供的StringBuilder中，
     * 避免创建中间字符串对象，提高性能。
     *
     * @param sb 目标StringBuilder
     * @param replacementTemplate 替换模板字符串
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see Matcher#expandReplacement(String)
     */
    void expandReplacementInto(StringBuilder sb, String replacementTemplate);

    /**
     * 预处理替换模板字符串
     * <p>
     * 展开替换模板字符串并缓存，调用此方法后，可通过getComputedReplacement()获取计算结果，
     * 使用完毕后可调用clearComputedReplacement()提前释放缓存。
     * <p>
     *
     * @param replacementTemplate 替换模板字符串
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see Matcher#expandReplacement(String)
     */
    void prepareReplacement(String replacementTemplate);

    /**
     * 获取经过预处理计算后的替换结果
     * <p>
     * 返回prepareReplacement()方法计算并缓存的替换文本。
     * 必须先调用prepareReplacement()方法进行预处理。
     *
     * @return 计算后的替换文本
     * @throws IllegalStateException 如果未调用prepareReplacement()进行预处理
     */
    String getComputedReplacement();

    /**
     * 清除已计算的替换结果缓存
     * <p>
     * 释放prepareReplacement()产生的计算结果占用的内存。
     * 调用此方法后，getComputedReplacement()将抛出IllegalStateException，
     * 需要重新调用prepareReplacement()。
     * <p>
     * 建议在不再需要缓存的替换结果时及时调用此方法以释放内存。
     */
    void clearComputedReplacement();

}
