package bin.mt.plugin.api.regex;

import java.util.regex.PatternSyntaxException;

/**
 * 正则表达式匹配器接口
 * 提供正则表达式匹配和查找功能
 */
public interface Matcher {

    /**
     * 获取当前匹配器关联的模式
     *
     * @return 当前匹配器的模式
     */
    Pattern pattern();

    /**
     * 获取当前匹配器关联的文本序列
     *
     * @return 当前匹配器的文本序列
     */
    CharSequence getText();

    /**
     * 重置匹配器状态
     *
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher reset();

    /**
     * 重置匹配器并设置新的输入文本
     *
     * @param text 新的输入文本序列
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher reset(CharSequence text);

    /**
     * 判断整个输入序列是否与正则表达式完全匹配
     *
     * @return 如果整个输入序列匹配则返回true，否则返回false
     */
    boolean matches();

    /**
     * 查找下一个匹配的子序列
     * <p>
     * 从上次匹配的结束位置开始查找，如果是第一次调用则从区域开始处查找。
     *
     * @return 如果找到匹配的子序列则返回true，否则返回false
     */
    boolean find();

    /**
     * 从指定位置开始查找下一个匹配的子序列
     *
     * @param start 开始查找的位置索引
     * @return 如果找到匹配的子序列则返回true，否则返回false
     */
    boolean find(int start);

    /**
     * 判断输入序列的开头是否与正则表达式匹配
     * <p>
     * 从区域的开头开始匹配，不要求匹配整个区域。
     *
     * @return 如果输入序列的开头匹配则返回true，否则返回false
     */
    boolean lookingAt();

    /**
     * 从指定位置开始判断输入序列是否与正则表达式匹配
     * <p>
     * 类似lookingAt()，但从指定位置开始匹配。
     *
     * @param start 开始匹配的位置索引
     * @return 如果从指定位置开始匹配则返回true，否则返回false
     */
    boolean lookingAt(int start);

    /**
     * 创建当前匹配状态的快照
     * <p>
     * 用于保存当前匹配结果，便于后续访问或恢复
     *
     * @return 匹配状态快照对象
     */
    MatcherSnapshot toSnapshot();

    /**
     * 获取上一次匹配操作中匹配子序列的起始位置
     * <p>
     * 等同于start(0)
     *
     * @return 匹配子序列的起始索引
     */
    int start();

    /**
     * 获取上一次匹配操作中指定捕获组的起始位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的起始索引，如果该组未参与匹配则返回-1
     */
    int start(int group);

    /**
     * 获取上一次匹配操作中匹配子序列的结束位置
     * <p>
     * 等同于end(0)
     *
     * @return 匹配子序列的结束索引（不包含该位置的字符）
     */
    int end();

    /**
     * 获取上一次匹配操作中指定捕获组的结束位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的结束索引（不包含该位置的字符），如果该组未参与匹配则返回-1
     */
    int end(int group);

    /**
     * 获取上一次匹配操作中匹配的子序列内容
     * <p>
     * 等同于group(0)
     *
     * @return 匹配的字符串内容
     */
    String group();

    /**
     * 获取上一次匹配操作中指定捕获组的内容
     *
     * @param group 捕获组的索引（0表示整个匹配，1及以上表示各个捕获组）
     * @return 指定捕获组的字符串内容，如果该组未参与匹配则返回null
     */
    String group(int group);

    /**
     * 获取正则表达式中捕获组的数量
     * <p>
     * 组0（整个匹配）不包含在计数中
     *
     * @return 捕获组的数量（不包括组0，即整个匹配）
     */
    int groupCount();

    /**
     * 设置正则表达式匹配的超时时间，默认2000毫秒
     * <p>
     * 用于防止复杂正则表达式造成的性能问题，超时后相关匹配方法会直接返回false
     *
     * @param timeoutMillis 超时时间（毫秒）
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher setTimeoutMillis(int timeoutMillis);

    /**
     * 获取当前设置的超时时间，默认2000毫秒
     *
     * @return 超时时间（毫秒）
     */
    int getTimeoutMillis();

    /**
     * 设置匹配器的匹配区域
     * <p>
     * 限制匹配操作只在指定的字符范围内进行
     *
     * @param start 区域起始位置（包含）
     * @param end   区域结束位置（不包含）
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher region(int start, int end);

    /**
     * 获取当前匹配区域的起始位置
     *
     * @return 匹配区域的起始索引
     */
    int regionStart();

    /**
     * 获取当前匹配区域的结束位置
     *
     * @return 匹配区域的结束索引
     */
    int regionEnd();

    /**
     * 检查是否启用了透明边界模式
     *
     * @return 如果启用透明边界则返回true，否则返回false
     * @see #useTransparentBounds(boolean)
     */
    boolean hasTransparentBounds();

    /**
     * 设置此匹配器区域边界的透明性
     *
     * <p> 透明边界：前瞻、后顾和边界匹配构造可以看到区域边界之外的内容
     * <p> 不透明边界：这些构造无法越过区域边界查看外部内容
     * <p> 默认使用不透明边界
     *
     * @param b true使用透明边界，false使用不透明边界
     * @return 此匹配器
     */
    Matcher useTransparentBounds(boolean b);

    /**
     * 检查是否启用了锚定边界模式
     * <p>
     * 锚定边界模式下，区域边界被视为输入序列的边界，
     * 锚点如^和$将匹配区域的边界。
     *
     * @return 如果启用锚定边界则返回true，否则返回false
     */
    boolean hasAnchoringBounds();

    /**
     * 设置锚定边界模式
     * <p>
     * 控制区域边界是否被视为输入序列的边界。
     * <p>
     * 默认情况下启用锚定边界，锚点如^和$将匹配区域的边界。
     * 禁用锚定边界后，锚点只匹配整个输入序列的真实边界。
     *
     * @param b true表示启用锚定边界，false表示禁用
     * @return 返回当前匹配器实例，支持链式调用
     */
    Matcher useAnchoringBounds(boolean b);

    /**
     * 检查上次匹配操作是否到达了输入序列的末尾
     * <p>
     * 如果匹配器在搜索过程中读取到了输入序列的最后一个字符，则返回true。
     * 这对于确定是否需要更多输入数据很有用。
     *
     * @return 如果匹配器到达输入末尾则返回true，否则返回false
     */
    boolean hitEnd();

    /**
     * 检查更多输入是否可能将正匹配变为负匹配
     * <p>
     * 如果此方法返回 true 且找到了匹配，则更多输入可能导致匹配丢失。
     * 如果此方法返回 false 且找到了匹配，则更多输入可能改变匹配但不会丢失匹配。
     * 如果没有找到匹配，则此方法无意义。
     *
     * @return 如果更多输入可能将正匹配变为负匹配则返回 true，否则返回 false
     */
    boolean requireEnd();

    /**
     * 展开替换模板字符串
     * <p>
     * 处理替换模板中的特殊字符和捕获组引用（如$1, $2等），
     * 返回展开后的实际替换文本。
     * <p>
     * 支持的引用格式：
     * <ul>
     * <li>{@code $n} 或 {@code ${n}}: 引用第n个捕获组</li>
     * <li>{@code ${name}}: 引用名称为name的捕获组</li>
     * </ul>
     * <p>
     * 支持的转义符号：
     * <ul>
     * <li>{@code \n}: 表示换行符</li>
     * <li>{@code \r}: 表示回车符</li>
     * <li>{@code \t}: 表示制表符</li>
     * <li>{@code \$}: 表示符号 $（用于输出字面量$而非捕获组引用）</li>
     * <li>{@code \\}: 表示符号 \（用于输出字面量反斜杠）</li>
     * </ul>
     * <p>
     * 支持的大小写转换：
     * <ul>
     * <li>{@code \l}: 将后续一个字符转为小写</li>
     * <li><code>&#92;u</code>: 将后续一个字符转为大写</li>
     * <li>{@code \L}: 将后续所有字符转为小写（直到字符串结束）</li>
     * <li>{@code \U}: 将后续所有字符转为大写（直到字符串结束）</li>
     * </ul>
     * <p>
     * 大小写转换使用示例：
     * <ul>
     * <li>{@code \L$0}: 将整个匹配文本转为小写</li>
     * <li><code>&#92;u$2</code>: 将第2个捕获组内容的第一个字符转为大写，其余字符保持不变</li>
     * <li><code>&#92;u\L$0</code>: 将整个匹配文本的第一个字符转为大写，其余字符转为小写</li>
     * <li>{@code \l\l${10}}: 将第10个捕获组内容的前两个字符转为小写，其余字符保持不变</li>
     * </ul>
     *
     * @param replacementTemplate 替换模板字符串
     * @return 展开后的替换文本
     * @throws PatternSyntaxException 如果替换模板格式无效
     */
    String expandReplacement(String replacementTemplate);

    /**
     * 将展开的替换内容追加到StringBuilder中
     * <p>
     * 功能同expandReplacement()，但直接追加到提供的StringBuilder中，
     * 避免创建中间字符串对象，提高性能。
     *
     * @param sb                  目标StringBuilder
     * @param replacementTemplate 替换模板字符串
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see #expandReplacement(String)
     */
    void expandReplacementInto(StringBuilder sb, String replacementTemplate);

    /**
     * 实现非终端追加和替换步骤
     * <p>
     * 将上次匹配之前的字符和替换文本追加到StringBuilder中。
     * 此方法与find()一起使用可以逐步处理和替换所有匹配。
     * <p>
     * 典型使用模式：
     * <pre>{@code
     * StringBuilder sb = new StringBuilder();
     * while (matcher.find()) {
     *     matcher.appendReplacement(sb, replacement);
     * }
     * matcher.appendTail(sb);
     * }</pre>
     *
     * @param sb          目标StringBuilder
     * @param replacement 替换字符串（可包含捕获组引用）
     * @return 返回当前匹配器实例，支持链式调用
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see #expandReplacement(String)
     */
    Matcher appendReplacement(StringBuilder sb, String replacement);

    /**
     * 实现终端追加和替换步骤
     * <p>
     * 将最后一次匹配之后的字符追加到StringBuilder中。
     * 通常在调用完所有appendReplacement()后调用此方法。
     *
     * @param sb 目标StringBuilder
     * @return 返回传入的StringBuilder
     */
    StringBuilder appendTail(StringBuilder sb);

    /**
     * 替换所有匹配的子序列
     * <p>
     * 将输入序列中所有匹配正则表达式的子序列替换为给定的替换字符串。
     * 等同于循环调用find()和appendReplacement()，最后调用appendTail()。
     *
     * @param replacement 替换字符串（可包含捕获组引用，如$1, $2等）
     * @return 替换后的完整字符串
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see #expandReplacement(String)
     */
    String replaceAll(String replacement);

    /**
     * 替换第一个匹配的子序列
     * <p>
     * 将输入序列中第一个匹配正则表达式的子序列替换为给定的替换字符串。
     *
     * @param replacement 替换字符串（可包含捕获组引用，如$1, $2等）
     * @return 替换后的完整字符串
     * @throws PatternSyntaxException 如果替换模板格式无效
     * @see #expandReplacement(String)
     */
    String replaceFirst(String replacement);
}
