package bin.mt.plugin.api.editor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.json.JSONObject;
import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;
import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 文本编辑器底部快捷功能扩展接口
 */
public interface TextEditorFunction {

    /**
     * 在{@link #getOptionsData(PluginUI, PluginView)}方法的实现中，如果输入内容验证失败可返回该值来
     * 中断数据保存操作，防止设置界面被关闭
     * <p>
     * 例如某个编辑框要求输入内容但用户没有输入，此时需要为编辑器请求焦点，弹出Toast提示用户输入内容，
     * 然后返回VALIDATION_FAILED
     */
    JSONObject VALIDATION_FAILED = JSONObject.unmodifiableObject(new JSONObject());

    /**
     * 初始化方法，在功能实例创建后立即调用
     * <p>
     * 此方法在每个实例的生命周期中仅调用一次，用于进行必要的初始化工作，
     * 如资源加载、配置读取、状态初始化等。
     *
     * @param context MT插件上下文，提供插件运行所需的环境和资源访问能力
     */
    void init(PluginContext context);

    /**
     * 获取插件上下文
     * <p>
     * 返回在init()方法中传入的PluginContext实例，用于后续操作中
     * 访问插件框架提供的各种服务和资源。
     *
     * @return 插件上下文实例，不会为null
     */
    @NonNull
    PluginContext getContext();

    /**
     * 判断当前快捷功能是否启用
     * <p>
     * <ul>
     * <li>返回 {@code false} 时：
     *     <ul>
     *     <li>用户在添加快捷功能时将无法看到此功能选项</li>
     *     <li>已添加的此快捷功能按钮将变为不可用状态</li>
     *     </ul>
     * </li>
     * <li>返回 {@code true} 时：功能正常显示和使用</li>
     * </ul>
     * <p>
     * 典型使用场景：
     * <ul>
     * <li><b>功能开关控制</b>：插件提供设置界面让用户选择启用或禁用某些功能</li>
     * <li><b>条件性启用</b>：根据运行环境、权限状态或其他条件动态决定功能可用性</li>
     * <li><b>精简功能列表</b>：当插件实现多个功能接口时，避免选项过多造成用户困扰</li>
     * </ul>
     * <p>
     * <b>注意</b>：此方法可能被频繁调用，建议避免在其中执行耗时操作。
     *
     * @return {@code true} 表示功能启用，{@code false} 表示功能禁用
     */
    boolean isEnabled();

    /**
     * 获取快捷功能的显示名称
     * <p>
     * 如果返回的名称为{key}格式，，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @return 功能名称，不能为null或空字符串
     */
    @NonNull
    String name();

    /**
     * 是否支持在搜索/替换输入框中使用
     * <p>
     * 文本编辑器除了主编辑区域外，还包含搜索和替换功能的输入框。此方法用于控制功能的作用范围：
     * <ul>
     * <li>返回 true：当焦点在搜索/替换输入框内时，{@link #doFunction} 将处理焦点所在输入框的内容</li>
     * <li>返回 false：无论焦点在何处，{@link #doFunction} 始终处理主编辑器的内容</li>
     * </ul>
     * <p>
     * 例如：光标移动、插入文本等操作通常适合在输入框中使用；而代码格式化、语法检查等操作则仅适用于主编辑器。
     *
     * @return true 支持在搜索/替换输入框中使用；false 仅在主编辑器中可用
     */
    boolean supportEditTextView();

    /**
     * 是否支持长按重复执行
     * <p>
     * 快捷功能支持设置长按时重复执行短按操作。由于并非所有功能都适合重复执行，
     * 需要根据功能特性进行判断：
     * <ul>
     * <li>不适合重复执行：如复制文本功能，多次复制与单次复制效果相同</li>
     * <li>适合重复执行：如删除或插入文本功能，长按可实现连续删除或插入</li>
     * </ul>
     *
     * @return true 支持长按重复执行；false 仅支持单次点击执行
     */
    boolean supportRepeat();

    /**
     * 创建设置选项视图
     * <p>
     * 当快捷功能需要用户配置参数时（例如：插入模板的内容、格式化选项等），
     * 通过此方法创建自定义的配置选项视图。用户完成配置并确认保存后，
     * 系统将调用 {@link #getOptionsData(PluginUI, PluginView)} 方法获取用户输入的数据。
     *
     * @param pluginUI 插件UI接口
     * @param data     上次保存的配置数据，首次调用时为null，可用于回显历史配置
     * @return 配置选项视图；如果该功能不需要用户参数配置，则返回null
     */
    @Nullable
    PluginView buildOptionsView(@NonNull PluginUI pluginUI, @Nullable JSONObject data);

    /**
     * 获取用户配置的选项数据
     * <p>
     * 如果{@link #buildOptionsView(PluginUI, JSONObject)}方法返回非null视图，在用户填写完参数并点击确定后会调用此方法。
     * 您需要从传入的设置选项视图中获取用户输入的配置数据，转换为JSONObject格式返回，插件系统会保存这些数据，
     * 并在执行doFunction()或下次编辑时作为参数传入。
     * <p>
     * 如果输入内容验证失败，例如某个选项输入格式错误，您可以为输入框获取焦点并弹出Toast告知用户错误原因，
     * 然后返回VALIDATION_FAILED，这样配置对话框将会继续停留而不是直接消失。
     *
     * @param pluginUI   插件UI接口
     * @param pluginView 用户配置界面视图，由 {@link #buildOptionsView(PluginUI, JSONObject)} 创建
     * @return 用户配置数据的JSON对象；验证失败时返回 {@link #VALIDATION_FAILED}；不需要保存配置时返回null
     * @see #VALIDATION_FAILED
     */
    @Nullable
    JSONObject getOptionsData(@NonNull PluginUI pluginUI, @NonNull PluginView pluginView);

    /**
     * 快捷功能的核心方法
     * <p>
     * 当用户点击快捷功能按钮时调用此方法，实现具体的功能逻辑。
     * 可以通过editor参数操作文本内容，通过data参数获取用户配置的选项数据。
     *
     * @param pluginUI 插件UI接口
     * @param editor   文本编辑器实例，提供文本操作接口
     * @param data     用户配置的选项数据，来自getOptionsData()方法的返回值，可能为null
     */
    void doFunction(@NonNull PluginUI pluginUI, @NonNull TextEditor editor, @Nullable JSONObject data);
}
