package bin.mt.plugin.api.editor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.Objects;

import bin.mt.json.JSONObject;
import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;

/**
 * 文本编辑器快捷功能的抽象基类
 * <p>
 * 该抽象类实现了{@link TextEditorFunction}接口，为快捷功能开发提供基础实现，
 * 简化了常见场景下的开发工作。对于不需要复杂配置界面的简单快捷功能，
 * 继承此类可以减少样板代码的编写。
 * <p>
 * 基类特性：
 * <ul>
 * <li>自动管理插件上下文的存储和访问</li>
 * <li>提供init()钩子方法供子类进行自定义初始化</li>
 * <li>为不需要配置界面的功能提供默认的空实现</li>
 * </ul>
 * <p>
 * 使用示例：
 * <pre>{@code
 * public class MyTextFunction extends BaseTextEditorFunction {
 *     @NonNull
 *     @Override
 *     public String name() {
 *         return "我的功能";
 *     }
 *
 *     @Override
 *     public boolean supportEditTextView() {
 *         return false;
 *     }
 *
 *     @Override
 *     public boolean supportRepeat() {
 *         return true;
 *     }
 *
 *     @Override
 *     public void doFunction(PluginUI pluginUI, TextEditor editor, JSONObject data) {
 *         // 实现功能逻辑
 *         editor.insertText(0, "Hello World");
 *     }
 * }
 * }</pre>
 */
public abstract class BaseTextEditorFunction implements TextEditorFunction {
    /**
     * 插件上下文实例
     * <p>
     * 在init()方法中由框架自动设置，用于访问插件运行环境提供的各种服务和资源。
     * 子类可通过getContext()方法获取此实例。
     */
    private PluginContext context;

    /**
     * 初始化方法的最终实现
     * <p>
     * 此方法被标记为final，确保上下文设置逻辑不会被子类意外覆盖。
     * 方法执行流程：
     * <ol>
     * <li>保存传入的插件上下文实例</li>
     * <li>调用无参的init()钩子方法供子类进行自定义初始化</li>
     * </ol>
     *
     * @param context MT插件上下文，提供插件运行所需的环境和资源访问能力
     */
    @Override
    public final void init(PluginContext context) {
        this.context = context;
        init();
    }

    /**
     * 子类自定义初始化钩子方法
     * <p>
     * 此方法在插件上下文设置完成后被调用，子类可以重写此方法进行自己的初始化逻辑，
     * 如加载配置文件、初始化内部状态、注册监听器等。
     * <p>
     * 默认实现为空，子类根据需要选择性重写。
     * <p>
     * 注意：此时getContext()已可正常使用。
     */
    protected void init() {
        // 默认空实现，子类可根据需要重写
    }

    /**
     * 获取插件上下文实例
     * <p>
     * 返回在init()方法中设置的PluginContext实例，保证不为null。
     * 如果在init()调用前访问此方法，可能会抛出空指针异常。
     *
     * @return 插件上下文实例，不会为null（init调用后）
     */
    @NonNull
    @Override
    public PluginContext getContext() {
        return Objects.requireNonNull(context);
    }

    /**
     * 判断当前快捷功能是否启用的默认实现
     * <p>
     * 基类提供的默认实现始终返回 {@code true}，表示功能默认处于启用状态。
     * 这适用于大多数常规快捷功能的场景。
     * <p>
     * 子类可以重写此方法以实现自定义的启用逻辑，重写示例：
     * <pre>{@code
     * @Override
     * public boolean isEnabled() {
     *     // 示例：根据用户设置决定是否启用
     *     return getContext().getPreferences().getBoolean("enable_my_function", true);
     * }
     * }</pre>
     *
     * @return 默认返回 {@code true}，表示功能启用
     */
    @Override
    public boolean isEnabled() {
        return true;
    }

    /**
     * 创建设置选项界面的默认实现
     * <p>
     * 基类提供的默认实现返回null，表示不提供配置界面。
     * 对于需要用户配置参数的快捷功能，子类应该重写此方法创建相应的UI界面。
     *
     * @param pluginUI 插件UI接口
     * @param data     上次保存的配置数据，首次调用时为null
     * @return 默认返回null，表示无需配置界面
     * @see TextEditorFunction#buildOptionsView(PluginUI, JSONObject)
     */
    @Nullable
    @Override
    public PluginView buildOptionsView(@NonNull PluginUI pluginUI, @Nullable JSONObject data) {
        return null;
    }

    /**
     * 获取用户配置选项数据的默认实现
     * <p>
     * 基类提供的默认实现返回null，表示无配置数据需要保存。
     * 对于有配置界面的功能，子类需要重写此方法以提取和验证用户输入的数据。
     *
     * @param pluginUI   插件UI接口
     * @param pluginView 用户配置界面视图，由buildOptionsView()创建
     * @return 默认返回null，表示无配置数据
     * @see TextEditorFunction#getOptionsData(PluginUI, PluginView)
     */
    @Nullable
    @Override
    public JSONObject getOptionsData(@NonNull PluginUI pluginUI, @NonNull PluginView pluginView) {
        return null;
    }
}
