/*******************************************************************************
 * Copyright (c) 2016 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;


/**
 * 解析器事件的处理器。此类的实例可以传递给 {@link JSONParser}。
 * 解析器在读取输入时会调用给定处理器的方法
 * <p>
 * 这些方法的默认实现不执行任何操作。子类可以只重写它们感兴趣的方法。
 * 它们可以在任何时候使用 <code>getLocation()</code> 访问解析器的当前字符位置。
 * <code>start*</code> 方法会在位置指向被解析元素的第一个字符时被调用。
 * <code>end*</code> 方法会在位置指向被解析元素最后一个字符之后的位置时被调用。示例：
 * </p>
 *
 * <pre>
 * ["lorem ipsum"]
 *  ^            ^
 *  startString  endString
 * </pre>
 * <p>
 * 构建解析后 JSON 对象表示的子类可以在 {@link #startArray()} 和 {@link #startObject()} 中
 * 返回任意处理器对象用于 JSON 数组和 JSON 对象。这些处理器对象会在该特定数组或对象的
 * 所有后续解析器事件中被提供。它们可用于跟踪 JSON 数组或对象的元素
 * </p>
 *
 * @param <A> 用于 JSON 数组的处理器类型
 * @param <O> 用于 JSON 对象的处理器类型
 * @see JSONParser
 */
public abstract class JSONHandler<A, O> {

    JSONParser parser;

    /**
     * 返回当前的解析器位置
     *
     * @return 当前的解析器位置
     */
    protected Location getLocation() {
        return parser.getLocation();
    }

    /**
     * 表示 JSON 输入中 <code>null</code> 字面值的开始。
     * 此方法会在读取字面值的第一个字符时被调用
     */
    public void startNull() {
    }

    /**
     * 表示 JSON 输入中 <code>null</code> 字面值的结束。
     * 此方法会在读取字面值的最后一个字符后被调用
     */
    public void endNull() {
    }

    /**
     * 表示 JSON 输入中布尔字面值（<code>true</code> 或 <code>false</code>）的开始。
     * 此方法会在读取字面值的第一个字符时被调用
     */
    public void startBoolean() {
    }

    /**
     * 表示 JSON 输入中布尔字面值（<code>true</code> 或 <code>false</code>）的结束。
     * 此方法会在读取字面值的最后一个字符后被调用
     *
     * @param value 解析后的布尔值
     */
    public void endBoolean(boolean value) {
    }

    /**
     * 表示 JSON 输入中字符串的开始。
     * 此方法会在读取起始双引号字符（<code>'&quot;'</code>）时被调用
     */
    public void startString() {
    }

    /**
     * 表示 JSON 输入中字符串的结束。
     * 此方法会在读取结束双引号字符（<code>'&quot;'</code>）后被调用
     *
     * @param string 解析后的字符串
     */
    public void endString(String string) {
    }

    /**
     * 表示 JSON 输入中数字的开始。
     * 此方法会在读取数字的第一个字符时被调用
     */
    public void startNumber() {
    }

    /**
     * 表示 JSON 输入中数字的结束。
     * 此方法会在读取数字的最后一个字符后被调用
     *
     * @param string 解析后的数字字符串
     */
    public void endNumber(String string) {
    }

    /**
     * 表示 JSON 输入中数组的开始。
     * 此方法会在读取起始方括号字符（<code>'['</code>）时被调用
     * <p>
     * 此方法可以返回一个对象来处理该数组的后续解析器事件。
     * 这个数组处理器会在该数组的所有 {@link #startArrayValue(Object) startArrayValue()}、
     * {@link #endArrayValue(Object) endArrayValue()} 和 {@link #endArray(Object) endArray()} 调用中被提供
     * </p>
     *
     * @return 此数组的处理器，如果不需要则返回 <code>null</code>
     */
    public A startArray() {
        return null;
    }

    /**
     * 表示 JSON 输入中数组的结束。
     * 此方法会在读取结束方括号字符（<code>']'</code>）后被调用
     *
     * @param array 从 {@link #startArray()} 返回的数组处理器，如果未提供则为 <code>null</code>
     */
    public void endArray(A array) {
    }

    /**
     * 表示 JSON 输入中数组元素的开始。
     * 此方法会在读取元素的第一个字符时被调用，就在调用特定元素类型的 <code>start</code> 方法
     * （{@link #startString()}、{@link #startNumber()} 等）之前
     *
     * @param array 从 {@link #startArray()} 返回的数组处理器，如果未提供则为 <code>null</code>
     */
    public void startArrayValue(A array) {
    }

    /**
     * 表示 JSON 输入中数组元素的结束。
     * 此方法会在读取元素值的最后一个字符后被调用，就在调用特定元素类型的 <code>end</code> 方法
     * （如 {@link #endString(String) endString()}、{@link #endNumber(String) endNumber()} 等）之后
     *
     * @param array 从 {@link #startArray()} 返回的数组处理器，如果未提供则为 <code>null</code>
     */
    public void endArrayValue(A array) {
    }

    /**
     * 表示 JSON 输入中对象的开始。
     * 此方法会在读取起始花括号字符（<code>'{'</code>）时被调用
     * <p>
     * 此方法可以返回一个对象来处理该对象的后续解析器事件。
     * 这个对象处理器会在该对象的所有 {@link #startObjectName(Object) startObjectName()}、
     * {@link #endObjectName(Object, String) endObjectName()}、
     * {@link #startObjectValue(Object, String) startObjectValue()}、
     * {@link #endObjectValue(Object, String) endObjectValue()} 和
     * {@link #endObject(Object) endObject()} 调用中被提供
     * </p>
     *
     * @return 此对象的处理器，如果不需要则返回 <code>null</code>
     */
    public O startObject() {
        return null;
    }

    /**
     * 表示 JSON 输入中对象的结束。
     * 此方法会在读取结束花括号字符（<code>'}'</code>）后被调用
     *
     * @param object 从 {@link #startObject()} 返回的对象处理器，如果未提供则为 null
     */
    public void endObject(O object) {
    }

    /**
     * 表示 JSON 输入中对象成员名称的开始。
     * 此方法会在读取成员名称的起始引号字符（'&quot;'）时被调用
     *
     * @param object 从 {@link #startObject()} 返回的对象处理器，如果未提供则为 <code>null</code>
     */
    public void startObjectName(O object) {
    }

    /**
     * 表示 JSON 输入中对象成员名称的结束。
     * 此方法会在读取成员名称的结束引号字符（<code>'"'</code>）后被调用
     *
     * @param object 从 {@link #startObject()} 返回的对象处理器，如果未提供则为 null
     * @param name   解析后的成员名称
     */
    public void endObjectName(O object, String name) {
    }

    /**
     * 表示 JSON 输入中对象成员值的开始。
     * 此方法会在读取成员值的第一个字符时被调用
     *
     * @param object 从 {@link #startObject()} 返回的对象处理器，如果未提供则为 <code>null</code>
     * @param name   成员名称
     */
    public void startObjectValue(O object, String name) {
    }

    /**
     * 表示 JSON 输入中对象成员值的结束。
     * 此方法会在读取成员值的最后一个字符后被调用，就在调用特定成员类型的 <code>end</code> 方法
     * （如 {@link #endString(String) endString()}、{@link #endNumber(String) endNumber()} 等）之后
     *
     * @param object 从 {@link #startObject()} 返回的对象处理器，如果未提供则为 null
     * @param name   解析后的成员名称
     */
    public void endObjectValue(O object, String name) {
    }

}
