/*******************************************************************************
 * Copyright (c) 2013, 2015 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;

import androidx.annotation.NonNull;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;


/**
 * 表示一个 JSON 数组，即 JSON 值的有序集合
 * <p>
 * 可以使用 <code>add(...)</code> 方法添加元素，这些方法接受 {@link JSONValue} 实例、
 * 字符串、基本数字类型和布尔值。要替换数组中的元素，请使用 <code>set(int, ...)</code> 方法
 * </p>
 * <p>
 * 可以通过索引使用 {@link #get(int)} 访问元素。此类还支持使用 {@link #iterator()} 或
 * 增强 for 循环按文档顺序遍历元素：
 * </p>
 * <pre>
 * for (JSONValue value : jsonArray) {
 *   ...
 * }
 * </pre>
 * <p>
 * 可以通过 {@link #values()} 方法获取等效的 {@link List}
 * </p>
 * <p>
 * 注意，此类<strong>不是线程安全的</strong>。如果多个线程同时访问一个 <code>JSONArray</code>
 * 实例，且其中至少有一个线程修改了数组内容，则必须在外部同步对该实例的访问。
 * 否则可能导致不一致的状态
 * </p>
 * <p>
 * 此类<strong>不应被客户端继承</strong>
 * </p>
 */
public class JSONArray extends JSONValue implements Iterable<JSONValue> {

    private final List<JSONValue> values;

    /**
     * 创建一个新的空 JSONArray
     */
    public JSONArray() {
        values = new ArrayList<>();
    }

    /**
     * 创建一个具有指定初始容量的新 JSONArray
     *
     * @param initialCapacity 数组的初始容量
     */
    public JSONArray(int initialCapacity) {
        values = new ArrayList<>(initialCapacity);
    }

    /**
     * 从给定的输入字符串解析创建一个新的 JSONArray
     */
    public JSONArray(String source) {
        this();
        if (source == null) {
            throw new NullPointerException("source is null");
        }
        JSON.DefaultHandler handler = new JSON.DefaultHandler();
        new JSONParser(handler).parseArray(this, source);
    }

    /**
     * 使用指定 JSON 数组的内容创建一个新的 JSONArray
     *
     * @param array 要获取初始内容的 JSONArray，不能为 <code>null</code>
     */
    public JSONArray(JSONArray array) {
        this(array, false);
    }

    private JSONArray(JSONArray array, boolean unmodifiable) {
        if (array == null) {
            throw new NullPointerException("array is null");
        }
        if (unmodifiable) {
            values = Collections.unmodifiableList(array.values);
        } else {
            values = new ArrayList<>(array.values);
        }
    }

    /**
     * 返回指定 JSONArray 的不可修改包装器。此方法允许提供对 JSONArray 的只读访问
     * <p>
     * 返回的 JSONArray 由给定数组支持并反映后续更改。
     * 尝试修改返回的 JSONArray 将导致 <code>UnsupportedOperationException</code>
     * </p>
     *
     * @param array 要返回不可修改 JSONArray 的 JSONArray
     * @return 指定 JSONArray 的不可修改视图
     */
    public static JSONArray unmodifiableArray(JSONArray array) {
        return new JSONArray(array, true);
    }

    /**
     * 将指定 <code>int</code> 值的 JSON 表示追加到此数组末尾
     *
     * @param value 要添加到数组的值
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(int value) {
        values.add(JSON.value(value));
        return this;
    }

    /**
     * 将指定 <code>long</code> 值的 JSON 表示追加到此数组末尾
     *
     * @param value 要添加到数组的值
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(long value) {
        values.add(JSON.value(value));
        return this;
    }

    /**
     * 将指定 <code>float</code> 值的 JSON 表示追加到此数组末尾
     *
     * @param value 要添加到数组的值
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(float value) {
        values.add(JSON.value(value));
        return this;
    }

    /**
     * 将指定 <code>double</code> 值的 JSON 表示追加到此数组末尾
     *
     * @param value 要添加到数组的值
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(double value) {
        values.add(JSON.value(value));
        return this;
    }

    /**
     * 将指定 <code>boolean</code> 值的 JSON 表示追加到此数组末尾
     *
     * @param value 要添加到数组的值
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(boolean value) {
        values.add(JSON.value(value));
        return this;
    }

    /**
     * 将指定字符串的 JSON 表示追加到此数组末尾
     *
     * @param value 要添加到数组的字符串
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(String value) {
        values.add(JSON.value(value));
        return this;
    }

    /**
     * 将指定的 JSON 值追加到此数组末尾
     *
     * @param value 要添加到数组的 JSONValue，不能为 <code>null</code>
     * @return 数组本身，以便链式调用
     */
    public JSONArray add(JSONValue value) {
        if (value == null) {
            throw new NullPointerException("value is null");
        }
        values.add(value);
        return this;
    }

    /**
     * 用指定 <code>int</code> 值的 JSON 表示替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的值
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, int value) {
        values.set(index, JSON.value(value));
        return this;
    }

    /**
     * 用指定 <code>long</code> 值的 JSON 表示替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的值
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, long value) {
        values.set(index, JSON.value(value));
        return this;
    }

    /**
     * 用指定 <code>float</code> 值的 JSON 表示替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的值
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, float value) {
        values.set(index, JSON.value(value));
        return this;
    }

    /**
     * 用指定 <code>double</code> 值的 JSON 表示替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的值
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, double value) {
        values.set(index, JSON.value(value));
        return this;
    }

    /**
     * 用指定 <code>boolean</code> 值的 JSON 表示替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的值
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, boolean value) {
        values.set(index, JSON.value(value));
        return this;
    }

    /**
     * 用指定字符串的 JSON 表示替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的字符串
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, String value) {
        values.set(index, JSON.value(value));
        return this;
    }

    /**
     * 用指定的 JSON 值替换此数组中指定位置的元素
     *
     * @param index 要替换的数组元素的索引
     * @param value 要存储在指定数组位置的值，不能为 <code>null</code>
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray set(int index, JSONValue value) {
        if (value == null) {
            throw new NullPointerException("value is null");
        }
        values.set(index, value);
        return this;
    }

    /**
     * 从此数组中移除指定索引处的元素
     *
     * @param index 要移除的元素的索引
     * @return 数组本身，以便链式调用
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONArray remove(int index) {
        values.remove(index);
        return this;
    }

    /**
     * 返回此数组中的元素数量
     *
     * @return 此数组中的元素数量
     */
    public int size() {
        return values.size();
    }

    /**
     * 如果此数组不包含任何元素则返回 <code>true</code>
     *
     * @return 如果此数组不包含任何元素则返回 <code>true</code>
     */
    public boolean isEmpty() {
        return values.isEmpty();
    }

    /**
     * 返回此数组中指定位置的元素值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置的元素值
     * @throws IndexOutOfBoundsException 如果索引超出范围，即 <code>index &lt; 0</code> 或
     *                                   <code>index &gt;= size</code>
     */
    public JSONValue get(int index) {
        return values.get(index);
    }

    /**
     * 返回此数组中指定位置的元素的 {@link JSONObject} 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 JSONObject 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public JSONObject getJSONObject(int index) {
        return get(index).asObject();
    }

    /**
     * 返回此数组中指定位置的元素的 {@link JSONArray} 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 JSONArray 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public JSONArray getJSONArray(int index) {
        return get(index).asArray();
    }

    /**
     * 返回此数组中指定位置的元素的 <code>int</code> 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 int 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public int getInt(int index) {
        return get(index).asInt();
    }

    /**
     * 返回此数组中指定位置的元素的 {@link Integer} 值。
     * 如果元素值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 Integer 值，如果元素为 null 则返回 <code>null</code>
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public Integer getIntegerObject(int index) {
        JSONValue value = get(index);
        return value != null && !value.isNull() ? value.asInt() : null;
    }

    /**
     * 返回此数组中指定位置的元素的 <code>long</code> 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 long 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public long getLong(int index) {
        return get(index).asLong();
    }

    /**
     * 返回此数组中指定位置的元素的 {@link Long} 值。
     * 如果元素值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 Long 值，如果元素为 null 则返回 <code>null</code>
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public Long getLongObject(int index) {
        JSONValue value = get(index);
        return value != null && !value.isNull() ? value.asLong() : null;
    }

    /**
     * 返回此数组中指定位置的元素的 <code>float</code> 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 float 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public float getFloat(int index) {
        return get(index).asFloat();
    }

    /**
     * 返回此数组中指定位置的元素的 {@link Float} 值。
     * 如果元素值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 Float 值，如果元素为 null 则返回 <code>null</code>
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public Float getFloatObject(int index) {
        JSONValue value = get(index);
        return value != null && !value.isNull() ? value.asFloat() : null;
    }

    /**
     * 返回此数组中指定位置的元素的 <code>double</code> 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 double 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public double getDouble(int index) {
        return get(index).asDouble();
    }

    /**
     * 返回此数组中指定位置的元素的 {@link Double} 值。
     * 如果元素值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 Double 值，如果元素为 null 则返回 <code>null</code>
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public Double getDoubleValue(int index) {
        JSONValue value = get(index);
        return value != null && !value.isNull() ? value.asDouble() : null;
    }

    /**
     * 返回此数组中指定位置的元素的 <code>boolean</code> 值
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 boolean 值
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public boolean getBoolean(int index) {
        return get(index).asBoolean();
    }

    /**
     * 返回此数组中指定位置的元素的 {@link Boolean} 值。
     * 如果元素值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 Boolean 值，如果元素为 null 则返回 <code>null</code>
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public Boolean getBooleanObject(int index) {
        JSONValue value = get(index);
        return value != null && !value.isNull() ? value.asBoolean() : null;
    }

    /**
     * 返回此数组中指定位置的元素的 <code>String</code> 值。
     * 如果元素值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param index 要返回的数组元素的索引
     * @return 指定位置元素的 String 值，如果元素为 null 则返回 <code>null</code>
     * @throws IndexOutOfBoundsException 如果索引超出范围
     */
    public String getString(int index) {
        JSONValue value = get(index);
        return value != null && !value.isNull() ? value.asString() : null;
    }

    /**
     * 按文档顺序返回此数组中值的列表。返回的列表由此数组支持并将反映后续更改。
     * 它不能用于修改此数组。尝试修改返回的列表将导致异常
     *
     * @return 此数组中值的列表
     */
    public List<JSONValue> values() {
        return Collections.unmodifiableList(values);
    }

    /**
     * 按文档顺序返回此数组中值的迭代器。返回的迭代器不能用于修改此数组
     *
     * @return 此数组中值的迭代器
     */
    @NonNull
    public Iterator<JSONValue> iterator() {
        final Iterator<JSONValue> iterator = values.iterator();
        return new Iterator<JSONValue>() {

            public boolean hasNext() {
                return iterator.hasNext();
            }

            public JSONValue next() {
                return iterator.next();
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    @Override
    void write(JSONWriter writer) throws IOException {
        writer.writeArrayOpen();
        Iterator<JSONValue> iterator = iterator();
        if (iterator.hasNext()) {
            iterator.next().write(writer);
            while (iterator.hasNext()) {
                writer.writeArraySeparator();
                iterator.next().write(writer);
            }
        }
        writer.writeArrayClose();
    }

    @Override
    public boolean isArray() {
        return true;
    }

    @Override
    public JSONArray asArray() {
        return this;
    }

    @Override
    public int hashCode() {
        return values.hashCode();
    }


    /**
     * 判断给定对象是否与此 JSONArray "相等"。如果对象也是 <code>JSONArray</code>
     * 且两个数组包含相同的值列表，则认为它们相等
     * <p>
     * 如果两个 JSONArray 相等，它们也将产生相同的 JSON 输出
     * </p>
     *
     * @param object 要与此 JSONArray 比较的对象
     * @return 如果指定对象与此 JSONArray 相等则返回 <tt>true</tt>，否则返回 <code>false</code>
     */
    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (object == null) {
            return false;
        }
        if (getClass() != object.getClass()) {
            return false;
        }
        JSONArray other = (JSONArray) object;
        return values.equals(other.values);
    }

}
