package bin.mt.plugin.api.ui.builder;

import bin.mt.plugin.api.ui.PluginLinearLayout;
import bin.mt.plugin.api.ui.PluginUI;

/**
 * 插件根布局构建器接口
 * <p>
 * 根布局构建器是插件UI构建的起始点，通过PluginUI.buildHorizontalLayout()或
 * PluginUI.buildVerticalLayout()创建，用于创建顶层容器布局。
 * </p>
 * <p>
 * 根布局通常是线性布局（水平或垂直方向），作为其他UI组件的容器。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * // 创建垂直根布局
 * PluginView view = pluginUI.buildVerticalLayout()
 *     .paddingDp(16, 16, 16, 16)
 *     // 上面的语句用于设置根布局的属性，必须放在addXXX方法前面
 *     .addTextView().text("标题").textSize(18);
 *     .addEditText().hint("请输入内容");
 *     .addButton().text("确认").onClick(this::onConfirm);
 *     .build();
 *
 * // 创建水平根布局
 * PluginView view = pluginUI.buildHorizontalLayout()
 *     .gravity(Gravity.CENTER_VERTICAL)
 *     // 上面的语句用于设置根布局的属性，必须放在addXXX方法前面
 *     .addTextView("label").text("选择：");
 *     .addSpinner("options").items(optionList);
 *     .build();
 * }</pre>
 * </p>
 *
 * @see PluginBaseViewBuilder
 * @see PluginLinearLayout
 * @see PluginUI#buildHorizontalLayout()
 * @see PluginUI#buildVerticalLayout()
 */
public interface PluginRootLayoutBuilder extends PluginBaseViewBuilder<PluginRootLayoutBuilder> {
    /**
     * 判断根布局是否为水平线性布局
     *
     * @return 是否为水平线性布局
     */
    boolean isHorizontalLayout();

    /**
     * 判断根布局是否为垂直线性布局
     *
     * @return 是否为垂直线性布局
     */
    boolean isVerticalLayout();

    /**
     * 判断根布局是否为帧布局
     *
     * @return 是否为帧布局
     */
    boolean isFrameLayout();

    /**
     * 设置布局的对齐方式
     * <p>
     * 对于垂直布局，可以设置子View的水平对齐（如Gravity.CENTER_HORIZONTAL、Gravity.LEFT等）
     * 对于水平布局，可以设置子View的垂直对齐（如Gravity.CENTER_VERTICAL、Gravity.TOP等）
     * 对于帧布局，不支持gravity()方法，会抛出RuntimeException异常
     * </p>
     * <p>
     * 常用的对齐方式属性值：
     * <ul>
     * <li>Gravity.CENTER - 居中对齐</li>
     * <li>Gravity.LEFT - 左对齐（水平方向）</li>
     * <li>Gravity.RIGHT - 右对齐（水平方向）</li>
     * <li>Gravity.TOP - 顶部对齐（垂直方向）</li>
     * <li>Gravity.BOTTOM - 底部对齐（垂直方向）</li>
     * <li>Gravity.CENTER_HORIZONTAL - 水平居中</li>
     * <li>Gravity.CENTER_VERTICAL - 垂直居中</li>
     * </ul>
     * </p>
     *
     * @param gravity 对齐方式，使用Gravity常量
     * @return 当前构建器实例，支持链式调用
     * @see android.view.Gravity
     */
    PluginRootLayoutBuilder gravity(int gravity);
}
