package bin.mt.plugin.api.ui.builder;

import androidx.annotation.Nullable;

import bin.mt.json.JSONObject;
import bin.mt.plugin.api.ui.PluginRadioButton;
import bin.mt.plugin.api.ui.PluginRadioGroup;

/**
 * 插件单选按钮组构建器接口
 * <p>
 * 单选按钮组在线性布局的基础上实现单选按钮的互斥功能，同一组的单选按钮只能有一个处于选中状态。
 * 当用户选择某个单选按钮时，同组的其他单选按钮会自动取消选中。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * pluginUIBuilder
 *     .addRadioGroup(true).children(builder -> builder
 *         .addRadioButton("radio1").text("选项1")
 *         .addRadioButton("radio2").text("选项2")
 *         .addRadioButton("radio3").text("选项3")
 *     ).check("radio1")
 *     ...
 * }</pre>
 * </p>
 *
 * @see PluginBaseLinearLayoutBuilder
 * @see PluginRadioGroup
 */
public interface PluginRadioGroupBuilder extends PluginBaseLinearLayoutBuilder<PluginRadioGroupBuilder> {

    /**
     * 设置默认选中的单选按钮
     *
     * @param id 要默认选中的单选按钮的ID，如果找不到对应ID的单选按钮，则全部不选中
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder check(@Nullable String id);

    /**
     * 从JSON数据中读取并设置选中ID
     * <p>
     * 该方法相当于调用 check(data.getString(id, null))，
     * 后续可通过{@link JSONObject#putCheckedId(PluginRadioGroup)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data JSON数据源，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder checkedId(@Nullable JSONObject data);

    /**
     * 从JSON数据中读取并设置选中ID，并指定默认值
     * <p>
     * 该方法相当于调用 check(data.getString(id, defaultValue))，
     * 后续可通过{@link JSONObject#putCheckedId(PluginRadioGroup)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data         JSON数据源，可以为null
     * @param defaultValue 当数据源为null或不包含对应字段时使用的默认值
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder checkedId(@Nullable JSONObject data, @Nullable String defaultValue);


    /**
     * 设置默认选中的单选按钮
     *
     * @param position 要默认选中的单选按钮的位置，从0开始，如果超出范围（比如-1）则全部按钮将取消选中
     * @return 当前构建器实例，支持链式调用
     * @see PluginRadioButton#getPositionInRadioGroup()
     */
    PluginRadioGroupBuilder check(int position);

    /**
     * 从JSON数据中读取并设置选中位置
     * <p>
     * 该方法相当于调用 check(data.getInt(id, -1))，
     * 后续可通过{@link JSONObject#putCheckedPosition(PluginRadioGroup)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data JSON数据源，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder checkedPosition(@Nullable JSONObject data);

    /**
     * 从JSON数据中读取并设置选中位置，并指定默认值
     * <p>
     * 该方法相当于调用 check(data.getInt(id, defaultValue))，
     * 后续可通过{@link JSONObject#putCheckedPosition(PluginRadioGroup)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data         JSON数据源，可以为null
     * @param defaultValue 当数据源为null或不包含对应字段时使用的默认值
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder checkedPosition(@Nullable JSONObject data, int defaultValue);

    /**
     * 设置选中状态改变监听器（首次创建完毕后不调用）
     * <p>
     * 使用此方法设置监听器时，在PluginRadioGroup首次创建完毕后，不会立刻调用监听器传入初始选中状态，
     * 仅在后续选中项发生改变时才会调用
     *
     * @param listener 选中状态改变监听器，传入null则移除当前监听器
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder postOnCheckedChanged(PluginRadioGroup.OnCheckedChangeListener listener);

    /**
     * 设置选中状态改变监听器（首次创建完毕后立刻调用）
     * <p>
     * 使用此方法设置监听器时，在PluginRadioGroup首次创建完毕后，立刻调用监听器传入初始选中状态
     *
     * @param listener 选中状态改变监听器，传入null则移除当前监听器
     * @return 当前构建器实例，支持链式调用
     */
    PluginRadioGroupBuilder onCheckedChanged(PluginRadioGroup.OnCheckedChangeListener listener);

}