package bin.mt.plugin.api.ui.builder;

import android.graphics.drawable.Drawable;
import android.text.InputType;

import androidx.annotation.Nullable;

import bin.mt.json.JSONObject;
import bin.mt.plugin.api.ui.PluginEditText;
import bin.mt.plugin.api.ui.PluginEditTextWatcher;
import bin.mt.plugin.api.ui.PluginView;

/**
 * 插件文本编辑框构建器接口
 * <p>
 * 文本编辑框是用户输入和编辑文本内容的交互组件，支持单行和多行输入、
 * 自动换行、语法高亮、文本监听等功能。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * pluginUIBuilder
 *     .addEditText("input_field") // 也可以是addEditBox()
 *     .text("初始内容")
 *     .hint("请输入内容...")
 *     .textSize(16)
 *     .maxLines(5)
 *     .inputTypeMultiline()
 *     .syntaxHighlight("Java")
 *     ...
 * }</pre>
 * </p>
 *
 * @see PluginBaseViewBuilder
 * @see PluginEditText
 */
public interface PluginEditTextBuilder extends PluginBaseViewBuilder<PluginEditTextBuilder> {

    /**
     * 设置编辑框的文本内容
     * <p>
     * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param text 要设置的文本内容
     * @return 当前构建器实例，支持链式调用
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    PluginEditTextBuilder text(CharSequence text);

    /**
     * 从JSON数据中读取文本内容
     * <p>
     * 该方法相当于调用 text(data.getString(id, null))，但不会尝试转化为本地化文本。
     * 后续可通过{@link JSONObject#putText(PluginEditText)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data JSON数据源，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder text(@Nullable JSONObject data);

    /**
     * 从JSON数据中读取文本内容，并指定默认值
     * <p>
     * 该方法相当于调用 text(data.getString(id, defaultValue))，但不会尝试转化为本地化文本。
     * 后续可通过{@link JSONObject#putText(PluginEditText)}写回数据
     * <p>
     * 注意：调用该方法前必须先通过id()方法设置组件ID。
     *
     * @param data         JSON数据源，可以为null
     * @param defaultValue 当数据源为null或不包含对应字段时使用的默认文本
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder text(@Nullable JSONObject data, String defaultValue);

    /**
     * 设置编辑框的提示文本，提示文本在编辑框为空时显示，用于指导用户输入
     * <p>
     * 如果hint类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param hint 提示文本内容，当编辑框为空时显示
     * @return 当前构建器实例，支持链式调用
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    PluginEditTextBuilder hint(CharSequence hint);

    /**
     * 设置文本字体大小
     *
     * @param size 字体大小，单位为sp
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder textSize(float size);

    /**
     * 设置是否为单行模式
     * <p>
     * <strong>单行模式特性：</strong>
     * <ul>
     * <li>输入换行符不会进行换行，实际效果类似于输入空格</li>
     * <li>建议关掉自动换行，不然字符占满宽度仍然会显示到下一行</li>
     * </ul>
     * </p>
     * <p>
     * <strong>设置 singleLine 和 inputType 会相互影响：</strong>
     * <ul>
     * <li>设置 singleLine 为 true：inputType 会被加上 TYPE_TEXT_FLAG_MULTI_LINE</li>
     * <li>设置 singleLine 为 false：inputType 会被去掉 TYPE_TEXT_FLAG_MULTI_LINE</li>
     * <li>设置 inputType 不包含 TYPE_TEXT_FLAG_MULTI_LINE：singleLine 会被改为 true</li>
     * <li>设置 inputType 包含 TYPE_TEXT_FLAG_MULTI_LINE：singleLine 会被改为 false</li>
     * </ul>
     * </p>
     * @param singleLine true设置为单行模式，false设置为多行模式
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder singleLine(boolean singleLine);

    /**
     * 设置固定行数
     * <p>
     * 编辑框高度将固定为指定行数，无论内容多少。
     *
     * @param lines 固定行数
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder lines(int lines);

    /**
     * 设置最小行数
     * <p>
     * 编辑框高度至少为指定行数，即使内容不足也会保持最小高度。
     * </p>
     *
     * @param minLines 最小行数
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder minLines(int minLines);

    /**
     * 设置最大行数
     * <p>
     * 编辑框高度最多为指定行数，超出部分可通过滚动查看。
     * </p>
     *
     * @param maxLines 最大行数
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder maxLines(int maxLines);

    /**
     * 设置最大输入字符数
     * <p>
     * 限制用户可输入的最大字符数量，超出限制后无法继续输入。
     * </p>
     *
     * @param maxLength 最大字符数，0或负数表示无限制
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder maxLength(int maxLength);

    /**
     * 设置输入类型
     * <p>
     * 控制软键盘的显示样式和输入限制，如文本、数字等。
     * 对于密码输入，目前还不支持隐藏密码，后续完善。
     * </p>
     * <strong>注意</strong>：设置 singleLine 和 inputType 会相互影响，具体请看 {@link #singleLine(boolean)} 的说明！
     *
     * @param inputType 输入类型，使用InputType常量
     * @return 当前构建器实例，支持链式调用
     *
     * @see InputType
     * @see InputType#TYPE_CLASS_TEXT
     * @see InputType#TYPE_CLASS_NUMBER
     */
    PluginEditTextBuilder inputType(int inputType);

    /**
     * 设置为多行输入模式
     * <p>
     * 该方法会自动配置输入类型为多行文本模式，等价于调用：
     * <code>inputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_FLAG_MULTI_LINE)</code>
     * </p>
     * <p>
     * 使用此模式时，软键盘将显示回车换行按键，允许用户输入多行文本。
     * </p>
     * <strong>注意</strong>：设置 singleLine 和 inputType 会相互影响，具体请看 {@link #singleLine(boolean)} 的说明！
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see InputType#TYPE_CLASS_TEXT
     * @see InputType#TYPE_TEXT_FLAG_MULTI_LINE
     */
    PluginEditTextBuilder inputTypeMultiline();

    /**
     * 设置为数字输入模式
     * <p>
     * 该方法会自动配置输入类型为纯数字模式，等价于调用：
     * <code>inputType(InputType.TYPE_CLASS_NUMBER)</code>
     * </p>
     * <p>
     * 使用此模式时，软键盘将只显示数字键盘，提升数字输入的用户体验。
     * </p>
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see InputType#TYPE_CLASS_NUMBER
     */
    PluginEditTextBuilder inputTypeNumber();

    /**
     * 禁用括号高亮功能
     * <p>
     * 默认情况下，当光标位于括号位置时会高亮显示对应的另一个括号。
     * 调用此方法可以禁用该功能。
     * </p>
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginEditText#setBracketHighlightEnable(boolean)
     */
    PluginEditTextBuilder disableBracketHighlight();

    /**
     * 设置光标位置
     *
     * @param selection 光标位置索引，从0开始
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder select(int selection);

    /**
     * 设置文本选择范围
     *
     * @param selectionStart 选择起始位置索引
     * @param selectionEnd   选择结束位置索引
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder select(int selectionStart, int selectionEnd);

    /**
     * 选中所有文本
     *
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder selectAll();

    /**
     * 将光标移动到文本末尾
     *
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder selectEnd();

    /**
     * 设置语法高亮
     * <p>
     * 为编辑框启用指定语言的语法高亮功能，如"Java"、"XML"、"JavaScript"等，
     * 也可以使用文件后缀名，如".java"、".xml"、".js"等，
     * 语法高亮可以提升代码编辑的体验，使不同语法元素以不同颜色显示。
     * </p>
     * <p>
     * 除了常规语法名称外，还可以使用以下内置语法常量：
     * <ul>
     * <li>{@link PluginEditText#SYNTAX_REGEX}：用于正则表达式搜索输入框，提供正则表达式语法高亮</li>
     * <li>{@link PluginEditText#SYNTAX_REGEX_REPLACEMENT}：用于正则表达式替换输入框，提供替换模式语法高亮（如$1、$2等引用）</li>
     * </ul>
     * </p>
     *
     * @param syntaxOrSuffix 语法名称或文件后缀名
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginEditText#SYNTAX_REGEX
     * @see PluginEditText#SYNTAX_REGEX_REPLACEMENT
     * @see <a href="https://mt2.cn/guide/file/mt-syntax.html#%E5%B1%9E%E6%80%A7-name">语法文件开发 - name</a>
     */
    PluginEditTextBuilder syntaxHighlight(String syntaxOrSuffix);

    /**
     * 设置软换行模式
     * <p>
     * 控制文本在达到边界时的换行行为：
     * <ul>
     * <li>SOFT_WRAP_DISABLE：禁用自动换行</li>
     * <li>SOFT_WRAP_KEEP_WORD：自动换行但尽量保持单词完整</li>
     * <li>SOFT_WRAP_COMPLETELY_FILLED：完全填充每一行</li>
     * </ul>
     * </p>
     *
     * @param softWrap 软换行模式，使用PluginEditText中的SOFT_WRAP_*常量
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginEditText#SOFT_WRAP_DISABLE
     * @see PluginEditText#SOFT_WRAP_KEEP_WORD
     * @see PluginEditText#SOFT_WRAP_COMPLETELY_FILLED
     */
    PluginEditTextBuilder softWrap(int softWrap);

    /**
     * 设置为只读模式
     *
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder readOnly();

    /**
     * 设置是否为只读模式
     *
     * @param readOnly 是否只读
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder readOnly(boolean readOnly);

    /**
     * 添加文本变化监听器
     * <p>
     * 监听编辑框文本内容的变化，包括用户输入、删除、替换等操作。
     * 每个回调方法都会额外传递当前的PluginEditText对象，方便在监听器中直接操作编辑框。
     * 可以添加多个监听器，它们会按添加顺序依次执行。
     *
     * @param textWatcher 文本变化监听器，用于监听文本内容的变化
     * @return 当前构建器实例，支持链式调用
     * @see PluginEditTextWatcher
     * @see PluginEditTextWatcher#beforeTextChanged(PluginEditText, CharSequence, int, int, int)
     * @see PluginEditTextWatcher#onTextChanged(PluginEditText, CharSequence, int, int, int)
     * @see PluginEditTextWatcher#afterTextChanged(PluginEditText, Editable)
     */
    PluginEditTextBuilder addTextChangedListener(PluginEditTextWatcher textWatcher);

    /**
     * 请求获取焦点并弹出输入法
     *
     * @return 当前构建器实例，支持链式调用
     */
    PluginEditTextBuilder requestFocusAndShowIME();

    /**
     * 判断是否为Box风格
     * <p>
     * Box风格下编辑框背景为圆角矩形，普通风格下编辑框底部为一条横线。
     * </p>
     *
     * @return true表示Box风格，false表示普通风格
     */
    boolean isBoxStyle();

    /**
     * @deprecated 编辑框不支持设置背景功能
     */
    @Override
    @Deprecated
    PluginEditTextBuilder background(Drawable background);

    /**
     * @deprecated 编辑框不支持设置背景颜色功能
     */
    @Override
    @Deprecated
    PluginEditTextBuilder backgroundColor(int color);

    /**
     * @deprecated 编辑框不支持点击监听器功能
     */
    @Override
    @Deprecated
    PluginEditTextBuilder onClick(PluginView.OnClickListener listener);

    /**
     * @deprecated 编辑框不支持长按监听器功能
     */
    @Override
    @Deprecated
    PluginEditTextBuilder onLongClick(PluginView.OnLongClickListener listener);
}
