package bin.mt.plugin.api.drawable;

import android.graphics.drawable.Drawable;

import java.io.InputStream;

import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.util.ServiceLoader;

/**
 * 矢量图加载器
 */
public abstract class VectorDrawableLoader {

    /**
     * 加载Android Vector格式的XML文件
     * <p>
     * 支持已编译的二进制XML文件和纯文本XML文件
     *
     * @param is 包含Vector XML内容的输入流
     * @return 矢量图标Drawable
     * @throws VectorParseException 当XML文件格式错误或解析失败时抛出
     */
    public static Drawable fromVectorXml(InputStream is) throws VectorParseException {
        return ServiceLoader.get(Impl.class).fromVectorXml(is);
    }

    /**
     * 从插件安装包的assets目录加载Android Vector格式的XML文件
     *
     * @param context    插件上下文，用于访问assets资源
     * @param assetsName assets目录下的XML文件名，例如："icons/my_icon.xml"
     * @return 矢量图标Drawable
     * @throws VectorParseException 当文件不存在、XML格式错误或解析失败时抛出
     */
    public static Drawable fromVectorXml(PluginContext context, String assetsName) throws VectorParseException {
        try (InputStream is = context.getAssetsAsStream(assetsName)) {
            return fromVectorXml(is);
        } catch (VectorParseException e) {
            throw e;
        } catch (Exception e) {
            throw new VectorParseException(e);
        }
    }

    /**
     * 加载SVG格式的矢量图形文件
     * <p>
     * 返回的图像尺寸将自动转换为dp单位，确保在不同密度的屏幕上保持一致的显示效果。
     * 例如：SVG原始尺寸为24x24像素时，返回的Drawable尺寸为24x24dp。
     *
     * @param is 包含SVG内容的输入流
     * @return 矢量图标Drawable
     * @throws VectorParseException 当SVG文件格式错误或解析失败时抛出
     */
    public static Drawable fromSvg(InputStream is) throws VectorParseException {
        return ServiceLoader.get(Impl.class).fromSvg(is);
    }

    /**
     * 从插件安装包的assets目录加载SVG格式的矢量图形文件
     * <p>
     * 返回的图像尺寸将自动转换为dp单位，确保在不同密度的屏幕上保持一致的显示效果。
     * 例如：SVG原始尺寸为24x24像素时，返回的Drawable尺寸为24x24dp。
     *
     * @param context 插件上下文，用于访问assets资源
     * @param assetsName assets目录下的SVG文件名，例如："icons/my_icon.svg"
     * @return 矢量图标Drawable
     * @throws VectorParseException 当文件不存在、SVG格式错误或解析失败时抛出
     */
    public static Drawable fromSvg(PluginContext context, String assetsName) throws VectorParseException {
        try (InputStream is = context.getAssetsAsStream(assetsName)) {
            return fromSvg(is);
        } catch (VectorParseException e) {
            throw e;
        } catch (Exception e) {
            throw new VectorParseException(e);
        }
    }

    private VectorDrawableLoader() {
    }

    interface Impl {
        Drawable fromVectorXml(InputStream is) throws VectorParseException;

        Drawable fromSvg(InputStream is) throws VectorParseException;
    }
}
