/*******************************************************************************
 * Copyright (c) 2015 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;

import java.io.IOException;
import java.io.Writer;


/**
 * 一个轻量级的写入缓冲区，用于减少对底层写入器执行的写入操作次数。
 * 此实现不是线程安全的。它有意偏离了 Writer 的约定。特别是，它不会刷新或
 * 关闭包装的写入器，也不确保包装的写入器处于打开状态
 */
class WritingBuffer extends Writer {

    private final Writer writer;
    private final char[] buffer;
    private int fill = 0;

    WritingBuffer(Writer writer) {
        this(writer, 16);
    }

    WritingBuffer(Writer writer, int bufferSize) {
        this.writer = writer;
        buffer = new char[bufferSize];
    }

    @Override
    public void write(int c) throws IOException {
        if (fill > buffer.length - 1) {
            flush();
        }
        buffer[fill++] = (char) c;
    }

    @Override
    public void write(char[] cbuf, int off, int len) throws IOException {
        if (fill > buffer.length - len) {
            flush();
            if (len > buffer.length) {
                writer.write(cbuf, off, len);
                return;
            }
        }
        System.arraycopy(cbuf, off, buffer, fill, len);
        fill += len;
    }

    @Override
    public void write(String str, int off, int len) throws IOException {
        if (fill > buffer.length - len) {
            flush();
            if (len > buffer.length) {
                writer.write(str, off, len);
                return;
            }
        }
        str.getChars(off, off + len, buffer, fill);
        fill += len;
    }

    /**
     * 刷新内部缓冲区，但不刷新包装的写入器
     */
    @Override
    public void flush() throws IOException {
        writer.write(buffer, 0, fill);
        fill = 0;
    }

    /**
     * 不关闭或刷新包装的写入器
     */
    @Override
    public void close() throws IOException {
    }

}
