/*******************************************************************************
 * Copyright (c) 2013, 2017 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;

import androidx.annotation.NonNull;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.io.Writer;


/**
 * 表示一个 JSON 值。可以是 JSON <strong>对象</strong>、<strong>数组</strong>、
 * <strong>数字</strong>、<strong>字符串</strong>，或者字面值
 * <strong>true</strong>、<strong>false</strong>、<strong>null</strong> 之一
 * <p>
 * 字面值 <strong>true</strong>、<strong>false</strong>、<strong>null</strong>
 * 由常量 {@link JSON#TRUE}、{@link JSON#FALSE}、{@link JSON#NULL} 表示
 * </p>
 * <p>
 * JSON <strong>对象</strong>和<strong>数组</strong>由子类型 {@link JSONObject} 和 {@link JSONArray} 表示。
 * 可以使用这些类的公共构造函数创建实例
 * </p>
 * <p>
 * 表示 JSON <strong>数字</strong>、<strong>字符串</strong>和<strong>布尔值</strong>的实例
 * 可以使用静态工厂方法 {@link JSON#value(String)}、{@link JSON#value(long)}、
 * {@link JSON#value(double)} 等创建
 * </p>
 * <p>
 * 要判断此类实例的类型，可以使用 {@link #isObject()}、{@link #isArray()}、
 * {@link #isString()}、{@link #isNumber()} 等方法
 * </p>
 * <p>
 * 如果已知 JSON 值的类型，可以使用 {@link #asObject()}、{@link #asArray()}、
 * {@link #asString()}、{@link #asInt()} 等方法直接获取相应类型的值
 * </p>
 * <p>
 * 此类<strong>不应被客户端继承</strong>
 * </p>
 */
public abstract class JSONValue implements Serializable {

    JSONValue() {
        // 阻止包外的子类化
    }

    /**
     * 判断此值是否表示 JSON 对象。如果是，则此值是 {@link JSONObject} 的实例
     *
     * @return 如果此值是 JSONObject 的实例则返回 <code>true</code>
     */
    public boolean isObject() {
        return false;
    }

    /**
     * 判断此值是否表示 JSON 数组。如果是，则此值是 {@link JSONArray} 的实例
     *
     * @return 如果此值是 JSONArray 的实例则返回 <code>true</code>
     */
    public boolean isArray() {
        return false;
    }

    /**
     * 判断此值是否表示 JSON 数字
     *
     * @return 如果此值表示 JSON 数字则返回 <code>true</code>
     */
    public boolean isNumber() {
        return false;
    }

    /**
     * 判断此值是否表示 JSON 字符串
     *
     * @return 如果此值表示 JSON 字符串则返回 <code>true</code>
     */
    public boolean isString() {
        return false;
    }

    /**
     * 判断此值是否表示布尔值
     *
     * @return 如果此值表示 JSON 字面值 <code>true</code> 或 <code>false</code> 则返回 <code>true</code>
     */
    public boolean isBoolean() {
        return false;
    }

    /**
     * 判断此值是否表示 JSON 字面值 <code>true</code>
     *
     * @return 如果此值表示 JSON 字面值 <code>true</code> 则返回 <code>true</code>
     */
    public boolean isTrue() {
        return false;
    }

    /**
     * 判断此值是否表示 JSON 字面值 <code>false</code>
     *
     * @return 如果此值表示 JSON 字面值 <code>false</code> 则返回 <code>true</code>
     */
    public boolean isFalse() {
        return false;
    }

    /**
     * 判断此值是否表示 JSON 字面值 <code>null</code>
     *
     * @return 如果此值表示 JSON 字面值 <code>null</code> 则返回 <code>true</code>
     */
    public boolean isNull() {
        return false;
    }

    /**
     * 将此 JSON 值转换为 {@link JSONObject}，前提是此值表示 JSON 对象。
     * 如果不是，则抛出异常
     *
     * @return 此值对应的 JSONObject
     * @throws UnsupportedOperationException 如果此值不是 JSON 对象
     */
    public JSONObject asObject() {
        throw new UnsupportedOperationException("Not an object: " + this);
    }

    /**
     * 将此 JSON 值转换为 {@link JSONArray}，前提是此值表示 JSON 数组。
     * 如果不是，则抛出异常
     *
     * @return 此值对应的 JSONArray
     * @throws UnsupportedOperationException 如果此值不是 JSON 数组
     */
    public JSONArray asArray() {
        throw new UnsupportedOperationException("Not an array: " + this);
    }

    /**
     * 将此 JSON 值转换为 <code>int</code> 值，前提是此值表示可解释为 Java <code>int</code> 的 JSON 数字。
     * 如果不是，则抛出异常
     * <p>
     * 要解释为 Java <code>int</code>，JSON 数字不能包含指数或小数部分。
     * 此外，数字必须在 <code>Integer</code> 范围内
     * </p>
     *
     * @return 此值转换为 <code>int</code>
     * @throws UnsupportedOperationException 如果此值不是 JSON 数字
     * @throws NumberFormatException         如果此 JSON 数字无法解释为 <code>int</code> 值
     */
    public int asInt() {
        throw new UnsupportedOperationException("Not a number: " + this);
    }

    /**
     * 将此 JSON 值转换为 <code>long</code> 值，前提是此值表示可解释为 Java <code>long</code> 的 JSON 数字。
     * 如果不是，则抛出异常
     * <p>
     * 要解释为 Java <code>long</code>，JSON 数字不能包含指数或小数部分。
     * 此外，数字必须在 <code>Long</code> 范围内
     * </p>
     *
     * @return 此值转换为 <code>long</code>
     * @throws UnsupportedOperationException 如果此值不是 JSON 数字
     * @throws NumberFormatException         如果此 JSON 数字无法解释为 <code>long</code> 值
     */
    public long asLong() {
        throw new UnsupportedOperationException("Not a number: " + this);
    }

    /**
     * 将此 JSON 值转换为 <code>float</code> 值，前提是此值表示 JSON 数字。
     * 如果不是，则抛出异常
     * <p>
     * 如果 JSON 数字超出 <code>Float</code> 范围，则返回 {@link Float#POSITIVE_INFINITY} 或
     * {@link Float#NEGATIVE_INFINITY}
     * </p>
     *
     * @return 此值转换为 <code>float</code>
     * @throws UnsupportedOperationException 如果此值不是 JSON 数字
     */
    public float asFloat() {
        throw new UnsupportedOperationException("Not a number: " + this);
    }

    /**
     * 将此 JSON 值转换为 <code>double</code> 值，前提是此值表示 JSON 数字。
     * 如果不是，则抛出异常
     * <p>
     * 如果 JSON 数字超出 <code>Double</code> 范围，则返回 {@link Double#POSITIVE_INFINITY} 或
     * {@link Double#NEGATIVE_INFINITY}
     * </p>
     *
     * @return 此值转换为 <code>double</code>
     * @throws UnsupportedOperationException 如果此值不是 JSON 数字
     */
    public double asDouble() {
        throw new UnsupportedOperationException("Not a number: " + this);
    }

    /**
     * 将此 JSON 值转换为 String，前提是此值表示 JSON 字符串。
     * 如果不是，则抛出异常
     *
     * @return 此值表示的字符串
     * @throws UnsupportedOperationException 如果此值不是 JSON 字符串
     */
    public String asString() {
        throw new UnsupportedOperationException("Not a string: " + this);
    }

    /**
     * 将此 JSON 值转换为 <code>boolean</code> 值，前提是此值是 <code>true</code> 或 <code>false</code>。
     * 如果不是，则抛出异常
     *
     * @return 此值转换为 <code>boolean</code>
     * @throws UnsupportedOperationException 如果此值既不是 <code>true</code> 也不是 <code>false</code>
     */
    public boolean asBoolean() {
        throw new UnsupportedOperationException("Not a boolean: " + this);
    }

    /**
     * 将此值的 JSON 表示以紧凑形式写入给定的写入器，不包含任何额外的空白字符
     * <p>
     * 使用 {@link java.io.BufferedWriter BufferedWriter} 可以提高写入性能
     * </p>
     *
     * @param writer 要写入此值的写入器
     * @throws IOException 如果写入器发生 I/O 错误
     */
    public void writeTo(Writer writer) throws IOException {
        writeTo(writer, WriterConfig.MINIMAL);
    }

    /**
     * 使用给定的格式化配置将此值的 JSON 表示写入给定的写入器
     * <p>
     * 使用 {@link java.io.BufferedWriter BufferedWriter} 可以提高写入性能
     * </p>
     *
     * @param writer 要写入此值的写入器
     * @param config 控制格式化的配置，或 <code>null</code> 表示紧凑形式
     * @throws IOException 如果写入器发生 I/O 错误
     */
    public void writeTo(Writer writer, WriterConfig config) throws IOException {
        if (writer == null) {
            throw new NullPointerException("writer is null");
        }
        if (config == null) {
            throw new NullPointerException("config is null");
        }
        WritingBuffer buffer = new WritingBuffer(writer, 128);
        write(config.createWriter(buffer));
        buffer.flush();
    }

    /**
     * 返回此值的紧凑形式 JSON 字符串，不包含任何额外的空白字符。
     * 结果保证是 {@link JSON#parse(String)} 方法的有效输入，
     * 并且会创建一个与此对象<em>相等</em>的值
     *
     * @return 表示此值的 JSON 字符串
     */
    @NonNull
    @Override
    public String toString() {
        return toString(WriterConfig.MINIMAL);
    }

    /**
     * 使用给定的格式化配置返回此值的 JSON 字符串
     *
     * @param config 控制格式化的配置，或 <code>null</code> 表示紧凑形式
     * @return 表示此值的 JSON 字符串
     */
    public String toString(WriterConfig config) {
        StringWriter writer = new StringWriter();
        try {
            writeTo(writer, config);
        } catch (IOException exception) {
            // StringWriter 不会抛出 IOException
            throw new RuntimeException(exception);
        }
        return writer.toString();
    }

    /**
     * 根据 {@link Object#equals(Object)} 中指定的约定，判断另一个对象是否与此对象"相等"
     * <p>
     * 当且仅当两个 JSONValue 表示相同的 JSON 文本时，才认为它们相等。
     * 因此，即使两个 JSONObject 包含相同的名称和值集合，
     * 但顺序不同，它们也可能不相等
     * </p>
     *
     * @param object 要与之比较的引用对象
     * @return 如果此对象与参数对象相同则返回 true；否则返回 false
     */
    @Override
    public boolean equals(Object object) {
        return super.equals(object);
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    abstract void write(JSONWriter writer) throws IOException;

}
