/*******************************************************************************
 * Copyright (c) 2015, 2016 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;

import java.io.IOException;
import java.io.Reader;


/**
 * 要<strong>解析</strong>给定的 JSON 输入，可以像下面的示例一样使用 <code>parse()</code> 方法：
 * <p>
 * <pre>
 * JSONObject object = JSON.parse(src).asObject();
 * JSONArray array = JSON.parse(src).asArray();
 * </pre>
 * <p>
 * 参数 <code>src</code> 可以是 <code>Reader</code> 或者 <code>String</code> 类型，
 * 当 <code>src</code> 是 <code>String</code> 类型时也可以使用以下方法创建：
 * <pre>
 * JSONObject object = new JSONObject(string);
 * JSONArray array = new JSONArray(string);
 * </pre>
 * <p>
 * 要<strong>创建</strong>用于序列化的 JSON 数据结构，请使用 <code>value()</code>、
 * <code>array()</code> 和 <code>object()</code> 方法。例如，以下代码片段将生成
 * JSON 字符串 <em>{"foo": 23, "bar": true}</em>：
 * </p>
 * <pre>
 * String string = JSON.object().add("foo", 23).add("bar", true).toString();
 * </pre>
 * <p>
 * 要从给定的 Java 数组创建 JSON 数组，可以使用带有可变参数的 <code>array()</code> 方法之一：
 * </p>
 * <pre>
 * String[] names = ...
 * JSONArray array = JSON.array(names);
 * </pre>
 */
public final class JSON {

    private JSON() {
        // 不应被实例化
    }

    /**
     * 表示 JSON 字面值 <code>null</code>
     */
    public static final JSONValue NULL = new JSONLiteral("null");

    /**
     * 表示 JSON 字面值 <code>true</code>
     */
    public static final JSONValue TRUE = new JSONLiteral("true");

    /**
     * 表示 JSON 字面值 <code>false</code>
     */
    public static final JSONValue FALSE = new JSONLiteral("false");

    /**
     * 返回表示给定 <code>int</code> 值的 JSONValue 实例
     *
     * @param value 要获取 JSON 表示形式的值
     * @return 表示给定值的 JSON 值
     */
    public static JSONValue value(int value) {
        return new JSONNumber(Integer.toString(value, 10));
    }

    /**
     * 返回表示给定 <code>long</code> 值的 JSONValue 实例
     *
     * @param value 要获取 JSON 表示形式的值
     * @return 表示给定值的 JSON 值
     */
    public static JSONValue value(long value) {
        return new JSONNumber(Long.toString(value, 10));
    }

    /**
     * 返回表示给定 <code>float</code> 值的 JSONValue 实例
     *
     * @param value 要获取 JSON 表示形式的值
     * @return 表示给定值的 JSON 值
     */
    public static JSONValue value(float value) {
        if (Float.isInfinite(value) || Float.isNaN(value)) {
            throw new IllegalArgumentException("Infinite and NaN values not permitted in JSON");
        }
        return new JSONNumber(cutOffPointZero(Float.toString(value)));
    }

    /**
     * 返回表示给定 <code>double</code> 值的 JSONValue 实例
     *
     * @param value 要获取 JSON 表示形式的值
     * @return 表示给定值的 JSON 值
     */
    public static JSONValue value(double value) {
        if (Double.isInfinite(value) || Double.isNaN(value)) {
            throw new IllegalArgumentException("Infinite and NaN values not permitted in JSON");
        }
        return new JSONNumber(cutOffPointZero(Double.toString(value)));
    }

    /**
     * 返回表示给定字符串的 JSONValue 实例
     *
     * @param string 要获取 JSON 表示形式的字符串
     * @return 表示给定字符串的 JSON 值
     */
    public static JSONValue value(String string) {
        return string == null ? NULL : new JSONString(string);
    }

    /**
     * 返回表示给定 <code>boolean</code> 值的 JSONValue 实例
     *
     * @param value 要获取 JSON 表示形式的值
     * @return 表示给定值的 JSON 值
     */
    public static JSONValue value(boolean value) {
        return value ? TRUE : FALSE;
    }

    /**
     * 创建一个新的空 JSONArray。这等同于使用构造函数创建新的 JSONArray
     *
     * @return 一个新的空 JSON 数组
     */
    public static JSONArray array() {
        return new JSONArray();
    }

    /**
     * 创建一个包含给定 <code>int</code> 值的 JSON 表示形式的新 JSONArray
     *
     * @param values 要包含在新 JSON 数组中的值
     * @return 包含给定值的新 JSON 数组
     */
    public static JSONArray array(int... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (int value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * 创建一个包含给定 <code>long</code> 值的 JSON 表示形式的新 JSONArray
     *
     * @param values 要包含在新 JSON 数组中的值
     * @return 包含给定值的新 JSON 数组
     */
    public static JSONArray array(long... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (long value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * 创建一个包含给定 <code>float</code> 值的 JSON 表示形式的新 JSONArray
     *
     * @param values 要包含在新 JSON 数组中的值
     * @return 包含给定值的新 JSON 数组
     */
    public static JSONArray array(float... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (float value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * 创建一个包含给定 <code>double</code> 值的 JSON 表示形式的新 JSONArray
     *
     * @param values 要包含在新 JSON 数组中的值
     * @return 包含给定值的新 JSON 数组
     */
    public static JSONArray array(double... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (double value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * 创建一个包含给定 <code>boolean</code> 值的 JSON 表示形式的新 JSONArray
     *
     * @param values 要包含在新 JSON 数组中的值
     * @return 包含给定值的新 JSON 数组
     */
    public static JSONArray array(boolean... values) {
        if (values == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (boolean value : values) {
            array.add(value);
        }
        return array;
    }

    /**
     * 创建一个包含给定字符串的 JSON 表示形式的新 JSONArray
     *
     * @param strings 要包含在新 JSON 数组中的字符串
     * @return 包含给定字符串的新 JSON 数组
     */
    public static JSONArray array(String... strings) {
        if (strings == null) {
            throw new NullPointerException("values is null");
        }
        JSONArray array = new JSONArray();
        for (String value : strings) {
            array.add(value);
        }
        return array;
    }

    /**
     * 创建一个新的空 JSONObject。这等同于使用构造函数创建新的 JSONObject
     *
     * @return 一个新的空 JSON 对象
     */
    public static JSONObject object() {
        return new JSONObject();
    }

    /**
     * 将给定的输入字符串解析为 JSON。输入必须包含有效的 JSON 值，可以在前后添加空白字符
     *
     * @param source 输入字符串，必须是有效的 JSON
     * @return 表示解析后 JSON 的值
     * @throws JSONParseException 如果输入不是有效的 JSON
     */
    public static JSONValue parse(String source) {
        if (source == null) {
            throw new NullPointerException("string is null");
        }
        DefaultHandler handler = new DefaultHandler();
        new JSONParser(handler).parse(source);
        return handler.getValue();
    }

    /**
     * 从给定的读取器中读取全部输入并解析为 JSON。输入必须包含有效的 JSON 值，
     * 可以在前后添加空白字符
     * <p>
     * 字符会被分块读入到输入缓冲区中。因此，将读取器包装在额外的
     * <code>BufferedReader</code> 中可能不会提高读取性能
     * </p>
     *
     * @param reader 用于读取 JSON 值的读取器
     * @return 表示解析后 JSON 的值
     * @throws IOException        如果读取器发生 I/O 错误
     * @throws JSONParseException 如果输入不是有效的 JSON
     */
    public static JSONValue parse(Reader reader) throws IOException {
        if (reader == null) {
            throw new NullPointerException("reader is null");
        }
        DefaultHandler handler = new DefaultHandler();
        new JSONParser(handler).parse(reader);
        return handler.getValue();
    }

    private static String cutOffPointZero(String string) {
        if (string.endsWith(".0")) {
            return string.substring(0, string.length() - 2);
        }
        return string;
    }

    static class DefaultHandler extends JSONHandler<JSONArray, JSONObject> {

        protected JSONValue value;

        @Override
        public JSONArray startArray() {
            return new JSONArray();
        }

        @Override
        public JSONObject startObject() {
            return new JSONObject();
        }

        @Override
        public void endNull() {
            value = NULL;
        }

        @Override
        public void endBoolean(boolean bool) {
            value = bool ? TRUE : FALSE;
        }

        @Override
        public void endString(String string) {
            value = new JSONString(string);
        }

        @Override
        public void endNumber(String string) {
            value = new JSONNumber(string);
        }

        @Override
        public void endArray(JSONArray array) {
            value = array;
        }

        @Override
        public void endObject(JSONObject object) {
            value = object;
        }

        @Override
        public void endArrayValue(JSONArray array) {
            array.add(value);
        }

        @Override
        public void endObjectValue(JSONObject object, String name) {
            object.add(name, value);
        }

        JSONValue getValue() {
            return value;
        }

    }

}
