package bin.mt.plugin.api.util;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.NonNull;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * UI更新器组管理类
 * <p>
 * 该类负责管理多个UIUpdater实例，并统一调度它们的待处理更新操作。
 * 主要功能：
 * - 注册和管理多个UIUpdater实例
 * - 通过Handler机制将更新操作调度到UI线程
 * - 使用延迟消息机制批量处理更新，提高性能
 * - 通过WeakReference避免内存泄漏
 * - 防止重复调度，确保效率
 */
public class UIUpdaterGroup {
    /**
     * 全局ID计数器，为每个UIUpdaterGroup实例分配唯一ID
     */
    private static final AtomicInteger ID_COUNTER = new AtomicInteger();

    /**
     * 全局共享的UI线程Handler
     * 用于将待处理的更新操作调度到主线程执行
     */
    private static final Handler HANDLER = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(@NonNull Message msg) {
            //noinspection unchecked
            WeakReference<UIUpdaterGroup> ref = (WeakReference<UIUpdaterGroup>) msg.obj;
            UIUpdaterGroup group = ref.get();
            if (group != null) {
                group.processPendingUpdates();
            }
        }
    };

    /**
     * 当前实例的唯一ID，用作Handler消息的标识符
     */
    private final int id = ID_COUNTER.getAndIncrement();

    /**
     * 指向自身的弱引用，用于Handler消息传递，避免Handler持有强引用导致内存泄漏
     */
    private final WeakReference<UIUpdaterGroup> myRef = new WeakReference<>(this);

    /**
     * 注册的UIUpdater列表，存储所有需要管理的更新器
     */
    private final List<UIUpdater<?>> updaters = new ArrayList<>();

    /**
     * 注册一个新的UI更新器
     * <p>
     * 创建一个新的UIUpdater实例并将其添加到管理列表中。
     *
     * @param consumer UI更新的消费者函数，定义具体的更新操作
     * @param <T>      更新数据的类型
     * @return 新创建的UIUpdater实例
     */
    public <T> UIUpdater<T> registerUpdater(Consumer<T> consumer) {
        UIUpdater<T> updater = new UIUpdater<>(this, consumer);
        updaters.add(updater);
        return updater;
    }

    /**
     * 调度待处理的更新操作
     */
    void schedulePendingUpdates() {
        if (!HANDLER.hasMessages(id)) {
            HANDLER.sendMessageDelayed(HANDLER.obtainMessage(id, myRef), 15);
        }
    }

    /**
     * 处理所有待处理的更新操作
     */
    private void processPendingUpdates() {
        // 遍历所有注册的更新器，处理它们的待处理更新
        for (int i = 0, size = updaters.size(); i < size; i++) {
            updaters.get(i).processPendingUpdate();
        }
    }
}
