package bin.mt.plugin.api.ui.menu;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 插件菜单接口
 * <p>
 * 提供菜单项和子菜单的管理功能，支持分组、查找、显示控制等操作
 */
public interface PluginMenu {

    /**
     * 添加一个菜单项，使用相同的值作为ID和标题
     * <p>
     * 相当于调用 add(idAndTitle.toString(), idAndTitle)
     *
     * @param idAndTitle 同时作为ID和标题的文本
     * @return 新添加的菜单项
     */
    PluginMenuItem add(@NonNull CharSequence idAndTitle);

    /**
     * 添加一个菜单项，指定ID和标题，不指定分组
     * <p>
     * 相当于调用 add(id, title, null)
     *
     * @param id    菜单项的唯一标识符
     * @param title 菜单项的标题
     * @return 新添加的菜单项
     */
    PluginMenuItem add(@NonNull String id, @NonNull CharSequence title);

    /**
     * 添加一个菜单项，指定ID、标题和分组
     * <p>
     * 如果title类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param id      菜单项的唯一标识符
     * @param title   菜单项的标题
     * @param groupId 菜单项所属的分组ID，null表示不分组
     * @return 新添加的菜单项
     */
    PluginMenuItem add(@NonNull String id, @NonNull CharSequence title, @Nullable String groupId);

    /**
     * 添加一个子菜单，使用相同的值作为ID和标题
     * <p>
     * 相当于调用 addSubMenu(idAndTitle.toString(), idAndTitle)
     *
     * @param idAndTitle 同时作为ID和标题的文本
     * @return 新添加的子菜单
     */
    PluginSubMenu addSubMenu(@NonNull CharSequence idAndTitle);

    /**
     * 添加一个子菜单，指定ID和标题，不指定分组
     * <p>
     * 相当于调用 addSubMenu(id, title, null)
     *
     * @param id    子菜单的唯一标识符
     * @param title 子菜单的标题
     * @return 新添加的子菜单
     */
    PluginSubMenu addSubMenu(@NonNull String id, @NonNull CharSequence title);

    /**
     * 添加一个子菜单，指定ID、标题和分组
     * <p>
     * 如果title类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param id      子菜单的唯一标识符
     * @param title   子菜单的标题
     * @param groupId 子菜单所属的分组ID，null表示不分组
     * @return 新添加的子菜单
     */
    PluginSubMenu addSubMenu(@NonNull String id, @NonNull CharSequence title, @Nullable String groupId);

    /**
     * 根据ID查找菜单项
     *
     * @param id 菜单项ID
     * @return 对应的菜单项，如果未找到则返回null
     */
    PluginMenuItem findItem(@NonNull String id);

    /**
     * 获取指定索引位置的菜单项
     *
     * @param index 菜单项索引
     * @return 对应索引的菜单项
     */
    PluginMenuItem getItem(int index);

    /**
     * 获取菜单项总数
     *
     * @return 菜单中的项目数量
     */
    int size();

    /**
     * 设置分组的可选中状态
     *
     * @param groupId   分组ID
     * @param checkable 是否可选中
     * @param exclusive 是否为互斥选择（单选模式）
     */
    void setGroupCheckable(@NonNull String groupId, boolean checkable, boolean exclusive);

    /**
     * 设置分组的可见性
     *
     * @param groupId 分组ID
     * @param visible 是否可见
     */
    void setGroupVisible(@NonNull String groupId, boolean visible);

    /**
     * 设置分组的启用状态
     *
     * @param groupId 分组ID
     * @param enabled 是否启用
     */
    void setGroupEnabled(@NonNull String groupId, boolean enabled);

    /**
     * 设置是否显示分组分隔线
     *
     * @param groupDividerEnabled 是否显示分隔线
     */
    void setGroupDividerEnabled(boolean groupDividerEnabled);

    /**
     * 清空菜单中的所有项目
     */
    void clear();

    /**
     * 菜单项点击监听器接口
     */
    interface OnMenuItemClickListener {
        /**
         * 菜单项被点击时的回调
         *
         * @param item 被点击的菜单项
         * @return true表示已处理该点击事件，false表示未处理
         */
        boolean onMenuItemClick(PluginMenuItem item);
    }

}
