package bin.mt.plugin.api.ui.dialog;

import android.view.KeyEvent;

import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.util.Consumer;

public abstract class BaseCancelableDialog<T extends BaseCancelableDialog<?>> {
    protected final PluginUI pluginUI;
    private boolean needSetKeyListener = true;
    private boolean cancelable;
    private boolean canceled;
    private Consumer<T> onCancelListener = BaseCancelableDialog::dismiss;

    public BaseCancelableDialog(PluginUI pluginUI) {
        this.pluginUI = pluginUI;
    }

    protected abstract PluginDialog getDialog();

    public abstract void dismiss();

    /**
     * 设置对话框为可取消状态，等同于调用setCancelable(true)
     *
     * @return 返回当前对话框实例
     */
    public T setCancelable() {
        return setCancelable(true);
    }

    /**
     * 设置对话框是否可以被取消
     * <p>
     * 当设置为可取消时，用户需要连续按两次返回键才能取消对话框
     *
     * @param cancelable true表示可取消，false表示不可取消
     * @return 返回当前对话框实例
     */
    public T setCancelable(boolean cancelable) {
        this.cancelable = cancelable;
        if (cancelable && needSetKeyListener) {
            needSetKeyListener = false;
            getDialog().setOnKeyListener(new PluginDialog.OnKeyListener() {
                long lastBackTime = 0;

                @Override
                public boolean onKey(PluginDialog dg, int keyCode, KeyEvent event) {
                    if (!BaseCancelableDialog.this.cancelable) {
                        return false;
                    }
                    if (event.getAction() == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_BACK) {
                        if (!canceled) {
                            if (System.currentTimeMillis() - lastBackTime < 2000) {
                                canceled = true;
                                if (onCancelListener != null) {
                                    //noinspection unchecked
                                    onCancelListener.accept((T) BaseCancelableDialog.this);
                                }
                            } else {
                                pluginUI.showToast("{press_again_to_cancel}");
                                lastBackTime = System.currentTimeMillis();
                            }
                        }
                        return true;
                    }
                    return false;
                }
            });
        }
        //noinspection unchecked
        return (T) this;
    }

    /**
     * 设置对话框取消时的回调监听器，默认监听器会直接调用dismiss()方法
     *
     * @param onCancelListener 取消监听器，接收当前对话框实例作为参数
     * @return 返回当前对话框实例
     */
    public T setOnCancelListener(Consumer<T> onCancelListener) {
        this.onCancelListener = onCancelListener;
        //noinspection unchecked
        return (T) this;
    }

    /**
     * 检查对话框是否已被取消
     *
     * @return true表示已取消，false表示未取消
     */
    public boolean isCanceled() {
        return canceled;
    }
}
