package bin.mt.plugin.api.ui.builder;

import android.content.res.ColorStateList;
import android.graphics.Bitmap;
import android.graphics.ColorFilter;
import android.graphics.Matrix;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.widget.ImageView;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.ui.PluginImageView;

/**
 * 插件图像视图构建器接口
 * <p>
 * 图像视图（ImageView）是用于显示图片的UI组件，支持多种图片源（Drawable、Bitmap等），
 * 提供丰富的图片显示和处理功能，包括缩放模式、色调、滤镜、透明度等设置。
 * </p>
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * // 显示Drawable图片
 * pluginUIBuilder
 *     .addImageView("avatar")
 *     .imageDrawable(drawable)
 *     .scaleType(ImageView.ScaleType.CENTER_CROP)
 *     .widthDp(48)
 *     .heightDp(48)
 *     ...
 *
 * // 显示Bitmap图片
 * pluginUIBuilder
 *     .addImageView("photo")
 *     .imageBitmap(bitmap)
 *     .scaleType(ImageView.ScaleType.FIT_CENTER)
 *     ...
 *
 * // 设置色调效果
 * pluginUIBuilder
 *     .addImageView("icon")
 *     .imageDrawable(drawable)
 *     .imageTintList(ColorStateList.valueOf(Color.BLUE))
 *     .imageTintMode(PorterDuff.Mode.SRC_IN)
 *     ...
 * }</pre>
 * </p>
 *
 * @see PluginBaseViewBuilder
 * @see PluginImageView
 */
public interface PluginImageViewBuilder extends PluginBaseViewBuilder<PluginImageViewBuilder> {

    /**
     * 设置要显示的图片Drawable对象
     * <p>
     * 用于显示矢量图、位图或其他类型的Drawable。
     * </p>
     *
     * @param drawable 要显示的图片Drawable对象
     * @return 当前构建器实例，支持链式调用
     */
    PluginImageViewBuilder image(@NonNull Drawable drawable);

    /**
     * 设置要显示的位图
     * <p>
     * 用于显示Bitmap类型的图片。
     * </p>
     *
     * @param bitmap 要显示的位图对象
     * @return 当前构建器实例，支持链式调用
     */
    PluginImageViewBuilder image(@NonNull Bitmap bitmap);

    /**
     * 设置图片色调列表
     * <p>
     * 通过ColorStateList为图片应用不同状态下的颜色叠加效果。
     * 色调会根据设置的混合模式与图片原色混合，实现图片颜色的动态变化。
     * </p>
     *
     * @param tint 色调列表
     * @return 当前构建器实例，支持链式调用
     * @see #imageTintMode(PorterDuff.Mode)
     */
    PluginImageViewBuilder imageTintList(@NonNull ColorStateList tint);

    /**
     * 设置图片色调的混合模式
     * <p>
     * 指定色调与图片原色的混合方式，不同的模式会产生不同的视觉效果。
     * 常用模式包括SRC_IN（仅保留色调颜色）、SRC_ATOP（保留图片形状）等。
     * </p>
     *
     * @param tintMode 色调混合模式
     * @return 当前构建器实例，支持链式调用
     * @see PorterDuff.Mode
     * @see #imageTintList(ColorStateList)
     */
    PluginImageViewBuilder imageTintMode(@NonNull PorterDuff.Mode tintMode);

    /**
     * 设置图片缩放类型
     * <p>
     * 控制图片如何在ImageView中缩放和定位。不同的缩放类型适用于不同的场景：
     * <ul>
     * <li>CENTER：图片居中显示，不缩放</li>
     * <li>CENTER_CROP：等比缩放，完全填充视图，可能裁剪图片</li>
     * <li>CENTER_INSIDE：等比缩放，完整显示图片，可能留白</li>
     * <li>FIT_CENTER：等比缩放居中，完整显示图片</li>
     * <li>FIT_XY：拉伸图片填充视图，可能变形</li>
     * </ul>
     * </p>
     *
     * @param scaleType 图片缩放类型
     * @return 当前构建器实例，支持链式调用
     * @see ImageView.ScaleType
     */
    PluginImageViewBuilder scaleType(@NonNull ImageView.ScaleType scaleType);

    /**
     * 设置图片的变换矩阵
     * <p>
     * 通过矩阵变换可以实现图片的任意缩放、旋转、平移、倾斜等效果。
     * 需要先将缩放类型设置为MATRIX此方法才会生效。
     * </p>
     *
     * @param matrix 图片变换矩阵
     * @return 当前构建器实例，支持链式调用
     * @see #scaleType(ImageView.ScaleType)
     * @see ImageView.ScaleType#MATRIX
     */
    PluginImageViewBuilder imageMatrix(@NonNull Matrix matrix);

    /**
     * 设置裁剪内边距
     * <p>
     * 启用后，图片将被裁剪到内边距范围内显示，
     * 超出内边距的部分不会显示。
     * </p>
     *
     * @param cropToPadding true表示启用裁剪内边距，false表示禁用
     * @return 当前构建器实例，支持链式调用
     */
    PluginImageViewBuilder cropToPadding(boolean cropToPadding);

    /**
     * 设置颜色滤镜
     * <p>
     * 使用指定的颜色和混合模式为图片应用颜色滤镜效果。
     * 滤镜会影响整个图片的颜色显示。
     * </p>
     *
     * @param color 滤镜颜色值（ARGB格式）
     * @param mode  颜色混合模式
     * @return 当前构建器实例，支持链式调用
     * @see PorterDuff.Mode
     */
    PluginImageViewBuilder colorFilter(int color, @NonNull PorterDuff.Mode mode);

    /**
     * 设置颜色滤镜
     * <p>
     * 使用指定颜色和默认混合模式（SRC_ATOP）为图片应用颜色滤镜效果。
     * </p>
     *
     * @param color 滤镜颜色值（ARGB格式）
     * @return 当前构建器实例，支持链式调用
     * @see #colorFilter(int, PorterDuff.Mode)
     */
    PluginImageViewBuilder colorFilter(int color);

    /**
     * 设置颜色滤镜
     * <p>
     * 使用ColorFilter对象为图片应用自定义的颜色滤镜效果。
     * ColorFilter支持多种滤镜类型，如ColorMatrixColorFilter、LightingColorFilter等。
     * </p>
     *
     * @param cf 颜色滤镜对象
     * @return 当前构建器实例，支持链式调用
     * @see ColorFilter
     */
    PluginImageViewBuilder colorFilter(@NonNull ColorFilter cf);

    /**
     * 设置图片的透明度
     * <p>
     * 设置图片的透明度，范围为0（完全透明）到255（完全不透明）。
     * 此方法仅影响图片的透明度，不影响ImageView本身及其背景。
     * </p>
     *
     * @param alpha 图片透明度值（0-255）
     * @return 当前构建器实例，支持链式调用
     */
    PluginImageViewBuilder imageAlpha(int alpha);
}

