package bin.mt.plugin.api.ui;

/**
 * 插件进度条视图接口
 * <p>
 * 进度条是一个用于显示操作进度或加载状态的UI组件。支持水平和圆形两种主要样式，
 * 可以显示确定进度（显示具体百分比）或不确定进度（显示动画但不显示具体进度）。
 * </p>
 *
 * @see bin.mt.plugin.api.ui.builder.PluginProgressBarBuilder
 */
public interface PluginProgressBar extends PluginView {

    /**
     * 获取当前进度条的样式（样式在构建时设置，构建完成后无法修改）
     *
     * @return 进度条样式枚举值
     * @see Style
     */
    Style getStyle();

    /**
     * 获取当前进度值
     *
     * @return 当前进度值，范围为0到maxProgress
     */
    int getProgress();

    /**
     * 设置当前进度值
     * <p>
     * 仅水平风格进度条可设置该选项，圆形进度条调用此方法无效果。
     * </p>
     *
     * @param progress 进度值，应在0到maxProgress范围内
     */
    void setProgress(int progress);

    /**
     * 获取次要进度值
     * <p>
     * 次要进度通常用于显示缓冲进度
     * </p>
     *
     * @return 次要进度值
     */
    int getSecondaryProgress();

    /**
     * 设置次要进度值
     * <p>
     * 仅水平风格进度条可设置该选项，圆形进度条调用此方法无效果。
     * 次要进度通常大于或等于当前进度，一般用于缓冲场景。
     * </p>
     *
     * @param secondaryProgress 次要进度值，应在0到maxProgress范围内
     */
    void setSecondaryProgress(int secondaryProgress);

    /**
     * 获取最大进度值
     *
     * @return 最大进度值，默认为100
     */
    int getMaxProgress();

    /**
     * 设置最大进度值
     * <p>
     * 仅水平风格进度条可设置该选项，圆形进度条调用此方法无效果。
     * 设置最大进度值后，当前进度和次要进度的有效范围为0到maxProgress。
     * </p>
     *
     * @param maxProgress 最大进度值，必须大于0
     */
    void setMaxProgress(int maxProgress);

    /**
     * 判断当前是否为不确定进度模式
     * <p>
     * 不确定进度模式下，进度条会显示循环动画，不显示具体进度百分比。
     * 适用于无法确定具体进度的长时间操作。
     * </p>
     *
     * @return true表示不确定进度模式，false表示确定进度模式
     */
    boolean isIndeterminate();

    /**
     * 设置是否为不确定进度模式
     * <p>
     * 仅水平风格进度条可设置该选项，圆形进度条默认为不确定进度模式。
     * 在不确定进度模式下，进度条会显示循环动画，忽略具体的进度值。
     * </p>
     *
     * @param indeterminate true设置为不确定进度模式，false设置为确定进度模式
     */
    void setIndeterminate(boolean indeterminate);

    /**
     * 进度条风格枚举
     */
    enum Style {
        /**
         * 水平进度条
         * <p>
         * 支持确定进度和不确定进度两种模式，可以设置具体的进度值、次要进度值和最大进度值。
         * </p>
         */
        HORIZONTAL,

        /**
         * 圆形进度条
         * <p>
         * 仅支持不确定进度模式，显示旋转动画。
         * </p>
         */
        CIRCULAR,

        /**
         * 小型圆形进度条
         * <p>
         * 仅支持不确定进度模式，显示旋转动画。尺寸较小，适用于空间受限的界面区域。
         * </p>
         */
        CIRCULAR_SMALL,

        /**
         * 大型圆形进度条
         * <p>
         * 仅支持不确定进度模式，显示旋转动画。尺寸较大，适用于需要突出显示加载状态的场景。
         * </p>
         */
        CIRCULAR_LARGE
    }
}
