package bin.mt.plugin.api.ui;

import android.graphics.drawable.Drawable;
import android.text.Editable;
import android.text.InputType;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 插件文本编辑框接口
 * 继承自PluginView，提供文本输入和编辑功能
 *
 * @see bin.mt.plugin.api.ui.builder.PluginEditTextBuilder
 */
public interface PluginEditText extends PluginView {
    /**
     * 内置语法名称，可用于正则表达式搜索输入框
     * <p>
     * 用法：<code>editText.setSyntaxHighlight(PluginEditText.SYNTAX_REGEX)</code>
     *
     * @see #setSyntaxHighlight(String)
     */
    String SYNTAX_REGEX = "INTERNAL:Regex";

    /**
     * 内置语法名称，可用于正则表达式替换输入框
     * <p>
     * 用法：<code>editText.setSyntaxHighlight(PluginEditText.SYNTAX_REGEX_REPLACEMENT)</code>
     *
     * @see #setSyntaxHighlight(String)
     */
    String SYNTAX_REGEX_REPLACEMENT = "INTERNAL:RegexReplacement";

    /**
     * 禁用自动换行模式
     * 文本超出边界时不会自动换行
     */
    int SOFT_WRAP_DISABLE = 0;

    /**
     * 自动换行-防止断词模式
     * 换行时尽量保持单词完整，避免在单词中间断开
     */
    int SOFT_WRAP_KEEP_WORD = 1;

    /**
     * 自动换行-完全填充模式
     * 完全填充每一行，不考虑单词边界
     */
    int SOFT_WRAP_COMPLETELY_FILLED = 2;

    /**
     * @return 编辑框中的文本内容
     */
    @NonNull
    Editable getText();

    /**
     * @return 编辑框中的文本长度
     */
    int length();

    /**
     * 设置编辑框的文本内容
     * <p>
     * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param text 要设置的文本内容
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    void setText(CharSequence text);

    /**
     * 获取编辑框的提示文本
     *
     * @return 提示文本内容
     */
    CharSequence getHint();

    /**
     * 设置编辑框的提示文本，提示文本在编辑框为空时显示，用于指导用户输入
     * <p>
     * 如果hint类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param hint 提示文本内容，当编辑框为空时显示
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    void setHint(CharSequence hint);

    /**
     * 获取文本字体大小
     *
     * @return 字体大小，单位为sp
     */
    float getTextSize();

    /**
     * 设置文本字体大小
     *
     * @param sizeSp 字体大小，单位为sp
     */
    void setTextSize(float sizeSp);

    /**
     * 判断是否为单行模式
     * <p>
     * 在单行模式下：
     * <ul>
     * <li>输入换行符不会进行换行，实际效果类似于输入空格</li>
     * <li>建议关掉自动换行，不然字符占满宽度仍然会显示到下一行</li>
     * <li>自动处理InputType，输入法的回车换行将会被禁用</li>
     * <li>如果设置InputType为多行模式，单行模式会被关闭</li>
     * </ul>
     * </p>
     *
     * @return true表示单行模式，false表示多行模式
     */
    boolean isSingleLine();

    /**
     * 设置是否为单行模式
     * <p>
     * <strong>单行模式特性：</strong>
     * <ul>
     * <li>输入换行符不会进行换行，实际效果类似于输入空格</li>
     * <li>建议关掉自动换行，不然字符占满宽度仍然会显示到下一行</li>
     * </ul>
     * </p>
     * <p>
     * <strong>设置 singleLine 和 inputType 会相互影响：</strong>
     * <ul>
     * <li>设置 singleLine 为 true：inputType 会被加上 TYPE_TEXT_FLAG_MULTI_LINE</li>
     * <li>设置 singleLine 为 false：inputType 会被去掉 TYPE_TEXT_FLAG_MULTI_LINE</li>
     * <li>设置 inputType 不包含 TYPE_TEXT_FLAG_MULTI_LINE：singleLine 会被改为 true</li>
     * <li>设置 inputType 包含 TYPE_TEXT_FLAG_MULTI_LINE：singleLine 会被改为 false</li>
     * </ul>
     * </p>
     * @param singleLine true设置为单行模式，false设置为多行模式
     */
    void setSingleLine(boolean singleLine);

    /**
     * 设置固定行数
     * <p>
     * 编辑框高度将固定为指定行数，无论内容多少。
     * </p>
     *
     * @param lines 固定行数
     */
    void setLines(int lines);

    /**
     * @return 最小行数
     */
    int getMinLines();

    /**
     * 设置最小行数
     * <p>
     * 编辑框高度至少为指定行数，即使内容不足也会保持最小高度。
     * </p>
     *
     * @param minLines 最小行数
     */
    void setMinLines(int minLines);

    /**
     * @return 最大行数
     */
    int getMaxLines();

    /**
     * 设置最大行数
     * <p>
     * 编辑框高度最多为指定行数，超出部分可通过滚动查看。
     * </p>
     *
     * @param maxLines 最大行数
     */
    void setMaxLines(int maxLines);

    /**
     * 设置最大输入字符数
     * <p>
     * 限制用户可输入的最大字符数量，超出限制后无法继续输入。
     * </p>
     *
     * @param maxLength 最大字符数，0或负数表示无限制
     */
    void setMaxLength(int maxLength);

    /**
     * 获取输入类型
     *
     * @return 输入类型常量值
     */
    int getInputType();

    /**
     * 设置输入类型
     * <p>
     * 控制软键盘的显示样式和输入限制，如文本、数字等。
     * </p>
     * <strong>注意</strong>：设置 singleLine 和 inputType 会相互影响，具体请看 {@link #setSingleLine(boolean)} 的说明！
     *
     * @param inputType 输入类型，使用InputType常量
     * @see InputType
     * @see InputType#TYPE_CLASS_TEXT
     * @see InputType#TYPE_CLASS_NUMBER
     */
    void setInputType(int inputType);

    /**
     * 设置为多行输入模式
     * <p>
     * 该方法会自动配置输入类型为多行文本模式，等价于调用：
     * <code>inputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_FLAG_MULTI_LINE)</code>
     * </p>
     * <p>
     * 使用此模式时，软键盘将显示回车换行按键，允许用户输入多行文本。
     * </p>
     * <strong>注意</strong>：设置 singleLine 和 inputType 会相互影响，具体请看 {@link #setSingleLine(boolean)} 的说明！
     *
     * @see InputType#TYPE_CLASS_TEXT
     * @see InputType#TYPE_TEXT_FLAG_MULTI_LINE
     */
    void setInputMultiline();

    /**
     * 设置为数字输入模式
     * <p>
     * 该方法会自动配置输入类型为纯数字模式，等价于调用：
     * <code>inputType(InputType.TYPE_CLASS_NUMBER)</code>
     * </p>
     * <p>
     * 使用此模式时，软键盘将只显示数字键盘，提升数字输入的用户体验。
     * </p>
     *
     * @see InputType#TYPE_CLASS_NUMBER
     */
    void setInputNumber();

    /**
     * 设置是否启用括号高亮功能
     * <p>
     * 启用括号高亮后，当光标所在位置是一个括号时，将高亮显示其对应的另一个括号。
     * 具体哪些括号会被高亮由编辑器当前使用的语法规则决定。
     *
     * @param enable true启用括号高亮，false禁用
     */
    void setBracketHighlightEnable(boolean enable);

    /**
     * 获取文本选择的起始位置，如果没有选择文本，selectionStart和selectionEnd均等于光标位置
     * <p>
     * 在某些情况下，selectionStart的值可能大于selectionEnd，一起使用时需要注意判断大小
     *
     * @return 选择起始位置的索引，
     */
    int getSelectionStart();

    /**
     * 获取文本选择的结束位置，如果没有选择文本，selectionStart和selectionEnd均等于光标位置
     * <p>
     * 注意，在某些情况下，selectionStart的值可能大于selectionEnd，一起使用时需要注意判断大小
     *
     * @return 选择结束位置的索引，注意！selectionStart可能大于selectionEnd
     */
    int getSelectionEnd();

    /**
     * 设置文本选择范围
     *
     * @param start 选择起始位置
     * @param end   选择结束位置
     */
    void setSelection(int start, int end);

    /**
     * 设置光标位置
     *
     * @param index 光标位置索引
     */
    void setSelection(int index);

    /**
     * 选中所有文本
     */
    void selectAll();

    /**
     * 将光标移动到文本末尾
     */
    void selectEnd();

    /**
     * 设置语法高亮
     * <p>
     * 为编辑框启用指定语言的语法高亮功能，如"Java"、"XML"、"JavaScript"等，
     * 也可以使用文件后缀名，如".java"、".xml"、".js"等，
     * 语法高亮可以提升代码编辑的体验，使不同语法元素以不同颜色显示。
     * </p>
     *
     * @param syntaxOrSuffix 语法名称或文件后缀名
     *
     * @see #SYNTAX_REGEX
     * @see #SYNTAX_REGEX_REPLACEMENT
     * @see <a href="https://mt2.cn/guide/file/mt-syntax.html#%E5%B1%9E%E6%80%A7-name">语法文件开发 - name</a>
     */
    void setSyntaxHighlight(@Nullable String syntaxOrSuffix);

    /**
     * 设置软换行模式
     * <p>
     * 控制文本在达到边界时的换行行为：
     * <ul>
     * <li>SOFT_WRAP_DISABLE：禁用自动换行</li>
     * <li>SOFT_WRAP_KEEP_WORD：自动换行但尽量保持单词完整</li>
     * <li>SOFT_WRAP_COMPLETELY_FILLED：完全填充每一行</li>
     * </ul>
     * </p>
     *
     * @param softWrap 软换行模式，使用PluginEditText中的SOFT_WRAP_*常量
     * @see PluginEditText#SOFT_WRAP_DISABLE
     * @see PluginEditText#SOFT_WRAP_KEEP_WORD
     * @see PluginEditText#SOFT_WRAP_COMPLETELY_FILLED
     */
    void setSoftWrap(int softWrap);

    /**
     * @return 是否只读模式
     */
    boolean isReadOnly();

    /**
     * 设置是否为只读模式
     *
     * @param readOnly 是否只读
     */
    void setReadOnly(boolean readOnly);

    /**
     * 判断是否为Box风格，普通风格下编辑框底部为一条横线，使用Box风格时编辑框背景为一个圆角矩形
     *
     * @return true表示Box风格，false表示普通风格
     */
    boolean isBoxStyle();

    /**
     * 添加文本变化监听器
     * <p>
     * 监听编辑框文本内容的变化，包括用户输入、删除、替换等操作。
     * 每个回调方法都会额外传递当前的PluginEditText对象，方便在监听器中直接操作编辑框。
     * 可以添加多个监听器，它们会按添加顺序依次执行。
     *
     * @param textWatcher 文本变化监听器，用于监听文本内容的变化
     * @see PluginEditTextWatcher
     * @see PluginEditTextWatcher#beforeTextChanged(PluginEditText, CharSequence, int, int, int)
     * @see PluginEditTextWatcher#onTextChanged(PluginEditText, CharSequence, int, int, int)
     * @see PluginEditTextWatcher#afterTextChanged(PluginEditText, Editable)
     */
    void addTextChangedListener(PluginEditTextWatcher textWatcher);

    /**
     * 请求获取焦点并弹出输入法
     *
     * @return 是否请求成功
     */
    boolean requestFocusAndShowIME();

    /**
     * @deprecated 编辑框不支持设置文本颜色，文本颜色由语法文件决定
     */
    @Deprecated
    void setTextColor(int textColor);

    /**
     * @deprecated 编辑框不支持获取背景功能
     */
    @Deprecated
    @Override
    Drawable getBackground();

    /**
     * @deprecated 编辑框不支持设置背景功能
     */
    @Deprecated
    @Override
    void setBackground(Drawable background);

    /**
     * @deprecated 编辑框不支持设置背景颜色功能
     */
    @Deprecated
    @Override
    void setBackgroundColor(int color);

    /**
     * @deprecated 编辑框不支持点击监听器功能
     */
    @Deprecated
    @Override
    void setOnClickListener(@Nullable OnClickListener listener);

    /**
     * @deprecated 编辑框不支持长按监听器功能
     */
    @Deprecated
    @Override
    void setOnLongClickListener(@Nullable OnLongClickListener listener);
}
