package bin.mt.plugin.api.translation;

import androidx.annotation.NonNull;

import java.io.IOException;

/**
 * 批量翻译引擎接口，扩展 {@link TranslationEngine} 以支持一次翻译多个文本。
 * <p>
 * 在翻译模式功能中，可能一次翻译数百上千个词条，对于需要网络请求的在线翻译引擎，批量翻译可以显著提升翻译效率。
 * 通过将多个文本合并为一次请求，可以减少网络往返次数，提高整体翻译速度。
 * <p>
 * <b>批量限制：</b>
 * 不同的翻译服务对单次请求有不同的限制。通过实现 {@link #createBatchingStrategy()} 方法返回自定义的
 * {@link BatchingStrategy}，引擎可以完全控制批次的拆分策略，支持基于词条数、字符数、字节数、Token 数等
 * 任意维度的限制条件。
 * <p>
 * <b>快速开始：</b>
 * 建议继承 {@link BaseBatchTranslationEngine} 而非直接实现此接口，基类提供了
 * {@link TranslationEngine#translate} 方法的默认实现，会自动委托给 {@link #batchTranslate}。
 *
 * @see TranslationEngine
 * @see BaseBatchTranslationEngine
 */
public interface BatchTranslationEngine extends TranslationEngine {

    /**
     * 批量执行文本翻译
     * <p>
     * 此方法运行在<b>子线程</b>，可能被<b>多次调用</b>以翻译不同批次的文本。
     * MT 会使用 {@link #createBatchingStrategy()} 返回的策略来拆分翻译任务。
     *
     * @param texts          待翻译的文本数组，不为 null 且不为空
     * @param sourceLanguage 源语言代码
     * @param targetLanguage 目标语言代码
     * @return 翻译后的文本数组，长度必须与输入数组相同，不能为 null
     * @throws IOException 翻译过程中发生的 IO 异常（如网络错误）
     */
    @NonNull
    String[] batchTranslate(String[] texts, String sourceLanguage, String targetLanguage) throws IOException;

    /**
     * 创建分批策略，用于控制翻译任务的拆分
     * <p>
     * 返回的 {@link BatchingStrategy} 实例可以维护内部状态（如累计字节数、Token 数等），
     * 实现增量计算以避免重复处理开销。
     * <p>
     * 实例会被复用，每次开始新批次时会调用 {@link BatchingStrategy#reset()}。
     *
     * @return 分批策略实例
     */
    BatchingStrategy createBatchingStrategy();

    /**
     * 分批策略，用于控制翻译任务的拆分
     * <p>
     * 实现类可以在内部维护任意状态（如累计字符数、字节数、Token 数等），
     * 在 {@link #tryAdd(String)} 时进行增量计算。
     */
    interface BatchingStrategy {
        /**
         * 重置状态，准备开始新批次
         * <p>
         * 此方法会在每个批次开始前被调用，实现类应清空所有累计状态。
         */
        void reset();

        /**
         * 尝试将文本加入当前批次
         * <p>
         * 实现类应根据自身的限制条件（如最大词条数、最大字符数、最大字节数等）
         * 判断是否可以容纳新文本。如果可以容纳，应更新内部状态并返回 {@code true}。
         * <p>
         * <b>MT 侧处理流程：</b>
         * <ul>
         *   <li>返回 {@code true}：将该文本加入当前批次</li>
         *   <li>返回 {@code false} 且当前批次不为空：翻译当前批次，调用 {@link #reset()} 开始新批次，
         *       再次使用该文本调用 {@code tryAdd} 尝试加入（即该文本会被调用两次）</li>
         *   <li>返回 {@code false} 且当前批次为空：强制将该文本加入当前批次并翻译，
         *       然后调用 {@link #reset()} 开始新批次（该文本仅调用一次）</li>
         * </ul>
         *
         * @param text 待加入的文本，不为 null
         * @return {@code true} 表示成功加入当前批次；{@code false} 表示当前批次已满
         */
        boolean tryAdd(String text);
    }

    /**
     * 默认分批策略实现，基于词条数和数据大小进行限制
     * <p>
     * 此类提供了一个通用的分批策略，可通过构造参数指定：
     * <ul>
     *   <li>{@code maxCount} - 单批次最大词条数</li>
     *   <li>{@code maxDataSize} - 单批次最大数据大小</li>
     * </ul>
     * <p>
     * 默认使用 {@link String#length()} 计算文本数据大小。子类可重写 {@link #getTextDataSize(String)}
     * 方法以实现自定义的计算逻辑（如按 UTF-8 字节数、Token 数等）。
     */
    class DefaultBatchingStrategy implements BatchingStrategy {
        private final int maxCount;
        private final int maxDataSize;
        private int count;
        private int totalDataSize;

        /**
         * 构造默认分批策略
         *
         * @param maxCount    单批次最大词条数，0 或负数表示无限制
         * @param maxDataSize 单批次最大数据大小，0 或负数表示无限制
         */
        public DefaultBatchingStrategy(int maxCount, int maxDataSize) {
            this.maxCount = maxCount;
            this.maxDataSize = maxDataSize;
        }

        @Override
        public void reset() {
            count = 0;
            totalDataSize = 0;
        }

        @Override
        public boolean tryAdd(String text) {
            if (maxCount > 0 && count >= maxCount) {
                return false;
            }
            int dataSize = getTextDataSize(text);
            if (maxDataSize > 0 && count > 0 && totalDataSize + dataSize > maxDataSize) {
                return false;
            }
            count++;
            totalDataSize += dataSize;
            return true;
        }

        /**
         * 计算文本的数据大小
         * <p>
         * 默认返回 {@link String#length()}。子类可重写此方法以实现自定义的计算逻辑，
         * 例如按 UTF-8 编码字节数：
         * <pre>{@code
         * @Override
         * protected int getTextDataSize(String text) {
         *     return text.getBytes(StandardCharsets.UTF_8).length;
         * }
         * }</pre>
         *
         * @param text 待计算的文本
         * @return 文本的数据大小
         */
        protected int getTextDataSize(String text) {
            return text.length();
        }
    }

}
