package bin.mt.plugin.api.editor;

import androidx.annotation.NonNull;

/**
 * 文本编辑器操作接口
 * 提供文本编辑、光标控制、选择操作等核心功能
 */
public interface TextEditor {

    ////////////////////////////////////////////////////////
    //
    // 光标与选中位置
    //
    ////////////////////////////////////////////////////////

    /**
     * 获取选中文本的起始位置。
     * <p>
     * 注意：此方法返回的是经过处理的起始位置，确保返回值始终小于等于结束位置。
     * 实际上返回的是原始selectionStart和selectionEnd中的较小值。
     * </p>
     * <p>
     * 当没有选中文本时（单选状态），此方法的返回值等于 {@link #getSelectionEnd()} 的返回值，
     * 表示光标所在的位置。
     * </p>
     *
     * @return 选中文本的起始位置，如果没有选中文本则返回光标位置
     */
    int getSelectionStart();

    /**
     * 获取选中文本的结束位置。
     * <p>
     * 注意：此方法返回的是经过处理的结束位置，确保返回值始终大于等于起始位置。
     * 实际上返回的是原始selectionStart和selectionEnd中的较大值。
     * </p>
     * <p>
     * 当没有选中文本时（单选状态），此方法的返回值等于 {@link #getSelectionStart()} 的返回值，
     * 表示光标所在的位置。
     * </p>
     *
     * @return 选中文本的结束位置，如果没有选中文本则返回光标位置
     */
    int getSelectionEnd();

    /**
     * 获取原始的选中文本起始位置。
     * <p>
     * 此方法返回的是真正的 Selection.getSelectionStart() 的值，不进行任何处理。
     * 在某些情况下，原始的selectionStart可能大于selectionEnd。
     * </p>
     * <p>
     * 当没有选中文本时（单选状态），此方法的返回值等于 {@link #getRawSelectionEnd()} 的返回值。
     * </p>
     *
     * @return 原始的选中文本起始位置
     */
    int getRawSelectionStart();

    /**
     * 获取原始的选中文本结束位置。
     * <p>
     * 此方法返回的是真正的 Selection.getSelectionEnd() 的值，不进行任何处理。
     * 在某些情况下，原始的selectionEnd可能小于selectionStart。
     * </p>
     * <p>
     * 当没有选中文本时（单选状态），此方法的返回值等于 {@link #getRawSelectionStart()} 的返回值。
     * </p>
     *
     * @return 原始的选中文本结束位置
     */
    int getRawSelectionEnd();

    /**
     * 判断当前是否选中了文本。
     * <p>
     * 此方法通过比较选中文本的起始位置和结束位置来判断是否有文本被选中：
     * <ul>
     * <li>当 {@link #getSelectionStart()} != {@link #getSelectionEnd()} 时，表示选中了文本</li>
     * <li>当 {@link #getSelectionStart()} == {@link #getSelectionEnd()} 时，表示只是光标定位（单选状态）</li>
     * </ul>
     * </p>
     *
     * @return 如果选中了文本返回true，如果只是光标定位返回false
     */
    boolean hasTextSelected();

    /**
     * 设置选中文本位置
     * <p>
     * 如果设置的选中位置在屏幕外，文本编辑器不会自动滚动，
     * 您可调用 {@link #ensureSelectionVisible()} 方法来滚动到选中位置
     *
     * @param selectionStart 选中文本的起始位置
     * @param selectionEnd   选中文本的结束位置
     */
    void setSelection(int selectionStart, int selectionEnd);

    /**
     * 设置光标位置
     * 等同于调用 setSelection(selection, selection)
     * <p>
     * 如果设置的光标位置在屏幕外，文本编辑器不会自动滚动，
     * 您可调用 {@link #ensureSelectionVisible()} 方法来滚动到光标位置
     *
     * @param selection 光标位置
     */
    void setSelection(int selection);

    /**
     * 显示光标
     * <p>
     * 光标会周期性隐藏和显示，调用该方法后会使光标立刻处于显示状态
     * <p>
     * 通常用于需要立即显示光标的场景，如用户交互后
     */
    void showCursor();

    /**
     * 将当前选中位置推送到撤销缓冲区
     * <ul>
     * <li><b>在修改文本前调用此方法</b>：点击撤销菜单，选中位置将恢复到调用此方法时的状态</li>
     * <li><b>在修改文本后调用此方法</b>：点击撤销菜单然后再点击重做菜单，选中位置将恢复到调用此方法时的状态</li>
     * <li>您可以先手动修改选中位置再调用此方法，这样就可以自定义撤销或重做后的选中位置</li>
     * </ul>
     * <strong>不调用此方法的情况：</strong>
     * <ul>
     * <li><b>插入文本然后撤销</b>：光标跳转到开始插入的位置</li>
     * <li><b>删除文本然后撤销</b>：选中被删除的文本</li>
     * <li><b>替换文本然后撤销</b>：选中被替换掉的文本</li>
     * <li><b>插入文本然后撤销再重做</b>：光标跳转到插入文本的末尾</li>
     * <li><b>删除文本然后撤销再重做</b>：光标跳转到删除点的位置</li>
     * <li><b>替换文本然后撤销再重做</b>：光标跳转新替换文本的末尾</li>
     * </ul>
     */
    void pushSelectionToUndoBuffer();

    ////////////////////////////////////////////////////////
    //
    // 位置可见相关
    //
    ////////////////////////////////////////////////////////

    /**
     * 确保指定位置在屏幕可见，如果不可见会自动进行跳转
     *
     * @param position    需要确保可见的文本位置索引
     * @param immediately 是否立即跳转，true时会立即跳转并禁用动画效果，false时使用平滑滚动动画
     */
    void ensurePositionVisible(int position, boolean immediately);

    /**
     * 确保指定位置在屏幕可见，如果不可见会自动进行跳转
     * 使用平滑滚动动画效果
     *
     * @param position 需要确保可见的文本位置索引
     */
    void ensurePositionVisible(int position);

    /**
     * 确保选中位置在屏幕可见，如果不可见会自动进行跳转
     * <ul>
     * <li>单选模式：确保光标位置在屏幕可见</li>
     * <li>多选模式：优先保证 selectionStart 在屏幕可见，并尽可能向 selectionEnd 所在位置滚动</li>
     * </ul>
     */
    void ensureSelectionVisible();

    ////////////////////////////////////////////////////////
    //
    // 文本操作
    //
    ////////////////////////////////////////////////////////

    /**
     * 获取文本编辑器的缓冲文本对象
     * <p>
     * 返回一个 {@link BufferedText} 实例，该实例提供了高效的文本访问和操作能力。
     * BufferedText 通过内置缓存机制优化了字符获取、文本搜索、行定位等操作的性能。
     * </p>
     *
     * <p>
     * <strong>注意事项：</strong>
     * <ul>
     *   <li>返回的 BufferedText 对象会缓存文本数据以提高性能</li>
     *   <li>当编辑器文本发生变化时，可能需要调用 {@link BufferedText#reset()} 来刷新缓存</li>
     * </ul>
     * </p>
     *
     * @return 文本编辑器的缓冲文本对象，永不为 null
     * @see BufferedText
     */
    @NonNull
    BufferedText getBufferedText();

    /**
     * 获取文本总长度
     *
     * @return 当前编辑器中文本的字符数量
     */
    int length();

    /**
     * 将指定范围的文本替换为指定文本的子串
     * 这是最基础的文本替换方法，其他文本操作方法都基于此实现
     *
     * @param start     要替换的文本起始位置
     * @param end       要替换的文本结束位置
     * @param text      用于替换的文本
     * @param textStart 替换文本的起始位置
     * @param textEnd   替换文本的结束位置
     */
    void replaceText(int start, int end, CharSequence text, int textStart, int textEnd);

    /**
     * 将指定范围的文本替换为新文本
     *
     * @param start 要替换的文本起始位置
     * @param end   要替换的文本结束位置
     * @param text  用于替换的完整文本
     */
    void replaceText(int start, int end, CharSequence text);

    /**
     * 向指定位置插入文本
     * 不会删除任何现有文本，仅在指定位置插入新文本
     *
     * @param position 插入位置
     * @param text     要插入的文本内容
     */
    void insertText(int position, CharSequence text);

    /**
     * 删除指定范围的文本
     *
     * @param start 删除范围的起始位置（包含）
     * @param end   删除范围的结束位置（不包含）
     */
    void deleteText(int start, int end);

    /**
     * 截取指定范围的文本
     *
     * @param start 起始位置（包含）
     * @param end   结束位置（不包含）
     * @return 截取的文本内容
     */
    String subText(int start, int end);

    /**
     * 获取当前选中的文本内容
     *
     * @return 当前选中的文本，如果没有选中文本（光标状态）则返回空字符串
     */
    String getSelectedText();

    ////////////////////////////////////////////////////////
    //
    // 大批量编辑模式
    //
    ////////////////////////////////////////////////////////

    /**
     * 开始大批量编辑模式
     * <p>
     * 由于每次修改文本时，编辑器都会对布局、代码高亮等进行计算与调整，如果在短时间内进行了多次修改，编辑器
     * 就会多次进行计算，计算量太大可能导致界面卡顿。（小文本正常感觉不出来，超大文本比较明显）
     * <p>
     * 因此如果实现某个功能需要多次修改文本，那么可调用 startLargeBatchEditingMode() 方法进入大批量编
     * 辑模式，此时编辑器会暂停所有耗时计算，等修改完成后再调用 finishLargeBatchEditingMode() 方法退出
     * 大批量编辑模式。
     * <p>
     * <b>重要提醒：</b>调用 startLargeBatchEditingMode() 后必须调用 finishLargeBatchEditingMode()，
     * 否则会导致异常，建议使用 try-finally 语法！
     *
     * <pre>{@code
     * textEditor.startLargeBatchEditingMode();
     * try {
     *     // 执行大量文本修改操作
     *     textEditor.insertText(0, "text1");
     *     textEditor.insertText(10, "text2");
     *     // ... 更多操作
     * } finally {
     *     textEditor.finishLargeBatchEditingMode();
     * }
     * }</pre>
     */
    void startLargeBatchEditingMode();

    /**
     * 结束大批量编辑模式
     * <p>
     * 必须与 startLargeBatchEditingMode() 配对使用
     * 调用后编辑器会恢复正常的计算和渲染，并对批量修改的内容进行统一处理
     */
    void finishLargeBatchEditingMode();

    ////////////////////////////////////////////////////////
    //
    // 杂项
    //
    ////////////////////////////////////////////////////////

    /**
     * 判断当前文本编辑器是否处于只读模式
     *
     * @return true表示只读，无法调用修改文本的相关方法。
     */
    boolean isReadOnly();

    /**
     * 获取设置中配置的缩进大小
     *
     * @return 缩进的字符数量，通常为2、4或8
     */
    int getTabSize();

    /**
     * 判断是否使用Tab字符进行缩进
     *
     * @return true表示使用\t字符进行缩进，false表示使用空格字符进行缩进
     */
    boolean isIndentWithTabs();

    /**
     * 获取光标处括号对的位置信息
     *
     * @return 成功时返回长度为2的int数组，[0]为第一个括号位置，[1]为第二个括号位置；
     * 失败时返回null（光标不在括号上或找不到匹配的括号）
     *
     * <p>括号对的介绍可查看 <a href="https://mt2.cn/guide/file/mt-syntax.html#%E5%B1%9E%E6%80%A7-bracketpairs">语法文件开发</a>
     */
    int[] getBracketPositions();

    /**
     * 获取光标处括号所对应的另一个括号的位置
     *
     * @return 匹配括号的位置索引，失败时返回-1（光标不在括号上或找不到匹配的括号）
     *
     * <p>括号对的介绍可查看 <a href="https://mt2.cn/guide/file/mt-syntax.html#%E5%B1%9E%E6%80%A7-bracketpairs">语法文件开发</a>
     */
    int getAnotherBracketPosition();

    /**
     * 显示浮动菜单
     * <p>
     * 浮动菜单将会自动选择一个合适的位置进行显示，尽量避免遮挡当前选中的文本内容。
     * 菜单的显示位置基于当前的文本选择范围自动计算。
     *
     * @see #showFloatingMenu(int)
     */
    void showFloatingMenu();

    /**
     * 在指定位置显示浮动菜单
     * <p>
     * 与无参方法 {@link #showFloatingMenu()} 不同，该方法允许手动指定浮动菜单的显示位置。
     * 菜单将显示在指定文本位置的附近。
     *
     * @param position 文本位置索引，指定菜单显示的参考位置。
     *                 例如可以传入 {@link #getSelectionEnd()} 在选区末尾显示菜单
     * @see #showFloatingMenu()
     */
    void showFloatingMenu(int position);

    /**
     * 获取文本编辑器当前的高亮语法名称
     *
     * @return 高亮语法名称，如果未开启高亮，则返回"Text"
     */
    String getSyntaxName();

    /**
     * 判断编辑框是否拥有焦点
     *
     * @return 是否拥有焦点
     */
    boolean isFocused();

    /**
     * 为编辑框请求获取焦点
     *
     * @return 是否成功获取焦点
     */
    boolean requestFocus();

    /**
     * 为编辑框请求获取焦点并弹出输入法
     *
     * @return 是否请求成功
     */
    boolean requestFocusAndShowIME();

    /**
     * 判断是否为输入框模式，MT内部将文本编辑器控件进行二次包装，并在许多界面中用作文本输入框去取代系统的 EditText，
     * 例如文本编辑器界面的搜索和替换输入框。
     *
     * @return 当前编辑器控件是否作为输入框使用
     */
    boolean isEditTextView();

}
