package bin.mt.plugin.api;

import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.File;
import java.io.InputStream;
import java.util.List;

import bin.mt.plugin.api.preference.PluginPreference;
import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * MT插件上下文接口
 * 提供插件运行时的核心功能和资源访问能力
 */
public interface PluginContext {
    /**
     * SDK版本号，当前版本为3
     */
    int SDK_VERSION = 3;

    /**
     * 获取当前插件的唯一标识符
     *
     * @return 插件ID字符串，例如"com.example.myplugin"
     */
    String getPluginId();

    /**
     * 获取当前插件名称
     *
     * @return 插件名称
     */
    String getPluginName();

    /**
     * 获取当前插件所使用的 SDK 版本号，该值在插件安装包的 manifest.json 文件中定义
     *
     * @return 插件 SDK 版本号
     */
    int getPluginSdkVersion();

    /**
     * 获取插件的版本代码（数字版本）
     * 用于版本比较和更新检测
     *
     * @return 插件版本代码，通常为递增的整数
     */
    int getPluginVersionCode();

    /**
     * 获取插件的版本名称（可读版本）
     * 用于向用户显示版本信息
     *
     * @return 插件版本名称，例如"1.0.0"、"2.1-beta"等
     */
    String getPluginVersionName();

    /**
     * 获取插件专用的配置存储对象
     * 可以自由进行增删查改操作，数据会持久化保存
     * 当插件被卸载时，相关数据会被自动删除
     *
     * @return SharedPreferences实例，用于存储插件配置数据
     */
    SharedPreferences getPreferences();

    /**
     * 获取当前系统的本地语言代码
     *
     * @return 语言代码，例如"zh"（中文）、"en"（英文）、"ru"（俄文）等
     */
    String getLanguage();

    /**
     * 获取当前系统的国家/地区代码
     *
     * @return 国家代码，例如"CN"（中国）、"US"（美国）等，可能为空字符串
     */
    String getCountry();

    /**
     * 获取语言和国家代码的组合字符串
     * 将Language和Country合并为标准的语言环境标识
     * <p>
     * <strong>示例：</strong>
     * <ul>
     * <li>Language为"zh"，Country为"CN"，返回"zh-CN"</li>
     * <li>Language为"en"，Country为空，返回"en"</li>
     * </ul>
     *
     * @return 组合后的语言环境标识
     */
    String getLanguageCountry();

    /**
     * 获取插件安装包assets文件夹内指定文件的输入流
     * 用于读取插件内置的资源文件
     *
     * @param name 文件相对路径，例如文件位于assets/data/config.txt，则参数为"data/config.txt"
     * @return 文件输入流，如果文件不存在则返回null
     */
    InputStream getAssetsAsStream(String name);

    /**
     * 根据key获取本地化文本，找不到对应的本地化文本返回null。
     * <p>
     * 关于key的详细介绍请参考 {@link #getString(String)}
     *
     * @param key 本地化文本的键值，不能为null或空字符串
     * @return key对应的本地化文本；如果所有语言包中都未找到，则返回null
     * @see #getString(String)
     */
    @Nullable
    String getStringNullable(@NonNull String key);

    /**
     * 根据key获取本地化文本
     * <p>
     * 支持的key格式：
     * <ol>
     *   <li>普通格式
     *     <ul>
     *       <li><b>"key"</b>：优先从strings语言包中获取对应词条，若未找到则回退到MT内置strings语言包中获取</li>
     *       <li><b>"pack:key"</b>：直接从指定的pack语言包中获取对应词条，没有回退机制</li>
     *     </ul>
     *   </li>
     *   <li>花括号格式（等价写法）
     *     <ul>
     *       <li><b>"{key}"</b>：等价于 "key"</li>
     *       <li><b>"{pack:key}"</b>：等价于 "pack:key"</li>
     *     </ul>
     *   </li>
     * </ol>
     * <p>
     * 使用示例：
     * <pre>{@code
     *   getString("plugin_name")     // 从strings包获取
     *   getString("common:ok")       // 从common包获取
     *   getString("{plugin_name}")   // 等价于第一种写法
     * }</pre>
     *
     * @param key 本地化文本的键值，不能为null或空字符串
     * @return 对应的本地化文本
     * @throws IllegalArgumentException 找不到与key对应的本地化文本
     */
    @NonNull
    String getString(@NonNull String key);


    /**
     * 根据key获取本地化文本并进行格式化
     * <p>
     * 获取本地化文本后，使用String.format()方法对文本进行格式化处理。
     * 关于key的详细介绍请参考 {@link #getString(String)}
     *
     * @param key        本地化文本的键值，支持多种格式
     * @param formatArgs 格式化参数，用于替换文本中的占位符（如%s, %d等）
     * @return 格式化后的本地化文本
     * @throws IllegalArgumentException         找不到与key对应的本地化文本
     * @throws java.util.IllegalFormatException 如果格式化参数与文本格式不匹配
     * @see String#format(String, Object...)
     * @see #getString(String)
     */
    String getString(@NonNull String key, Object... formatArgs);

    /**
     * 批量获取多个本地化文本
     * <p>
     * 按照传入的keys顺序，依次获取对应的本地化文本并返回List集合。
     * 关于key的详细介绍请参考 {@link #getString(String)}
     *
     * @param keys 本地化文本的键值数组
     * @return 包含所有本地化文本的List集合，顺序与传入的keys一致
     * @throws IllegalArgumentException 如果某个key未找到对应本地化文本
     * @see #getString(String)
     */
    List<String> getStringList(@NonNull String... keys);

    /**
     * 批量获取多个本地化文本并返回数组
     * <p>
     * 按照传入的keys顺序，依次获取对应的本地化文本并返回String数组。
     * 关于key的详细介绍请参考 {@link #getString(String)}
     *
     * @param keys 本地化文本的键值数组
     * @return 包含所有本地化文本的String数组，顺序与传入的keys一致
     * @throws IllegalArgumentException 如果某个key未找到对应本地化文本
     * @see #getString(String)
     */
    String[] getStringArray(@NonNull String... keys);

    /**
     * 相当于调用getAssetLocalString("String")
     *
     * @return LocalString实例，用于获取本地化文本
     * @deprecated 建议使用 {@link #getString(String)} 简化调用逻辑，而非手动管理语言包
     */
    @NonNull
    @Deprecated
    LocalString getLocalString();

    /**
     * 从插件安装包assets文件夹中获取指定名称的语言包
     *
     * @param name 语言包名称，对应assets目录下的语言包文件
     * @return LocalString实例，用于访问该语言包中的文本
     * @deprecated 建议使用 {@link #getString(String)} 简化调用逻辑，而非手动管理语言包
     */
    @NonNull
    @Deprecated
    LocalString getAssetLocalString(String name);

    /**
     * 从指定目录获取语言包文件
     *
     * @param dir  语言包所在的目录
     * @param name 语言包文件名称
     * @return LocalString实例，用于访问该语言包中的文本
     * @deprecated 建议使用 {@link #getString(String)} 简化调用逻辑，而非手动管理语言包
     */
    @NonNull
    @Deprecated
    LocalString getFileLocalString(File dir, String name);

    /**
     * 获取MT内置的全球语言名称本地化对象
     * 主要用于翻译引擎中显示语言名称
     * <p>
     * 支持的语言代码映射：
     * - "auto" -> "自动检测"
     * - "en" -> "英文"
     * - "zh-CN" -> "简体中文"
     * - 更多映射关系请查看MT安装包内的assets/lang-zh-CN.txt
     *
     * @return LocalString实例，包含各种语言的本地化名称
     * @deprecated 建议使用 {@link #getString(String)} 简化调用逻辑，而非手动管理语言包
     * @see bin.mt.plugin.api.translation.BaseTranslationEngine#getLanguageDisplayName(String)
     */
    @NonNull
    @Deprecated
    LocalString getLanguageNameLocalString();

    /**
     * 获取插件专用的文件存储目录
     * 该目录会在插件卸载时被自动删除，建议将所有插件文件存储在此目录
     * 可以有效防止卸载后产生残留文件
     *
     * @return 插件专用文件目录的File对象
     */
    File getFilesDir();

    /**
     * 跳转并显示设置界面
     *
     * @param clazz 设置界面类，需实现PluginPreference接口，为null则表示主设置界面类
     * @see PluginPreference
     */
    void openPreference(@Nullable Class<? extends PluginPreference> clazz);

    /**
     * 调用系统外部浏览器打开指定URL
     *
     * @param url 要跳转的URL地址，应为有效的HTTP/HTTPS链接
     */
    void openBrowser(String url);

    /**
     * 调用MT内置浏览器打开指定URL
     * <p>
     * 内置浏览器有个特点是不论跳转了多少页面，按返回键都是直接退出浏览器，不会返回上一级网页。
     * 适合仅展示信息，没有多层页面跳转的场景。
     *
     * @param url          要跳转的URL地址，应为有效的HTTP/HTTPS链接
     * @param showTitleBar 浏览器界面是否显示顶部标题栏
     */
    void openBuiltinBrowser(String url, boolean showTitleBar);

    /**
     * 判断剪贴板内是否有文本内容
     *
     * @return true表示剪贴板中有可用的文本内容，false表示剪贴板为空或不包含文本
     */
    boolean hasClipboardText();

    /**
     * 获取剪贴板中的文本内容
     *
     * @return 剪贴板中的文本，如果获取失败或剪贴板为空则返回空字符串（非null）
     */
    @NonNull
    CharSequence getClipboardText();

    /**
     * 将文本设置到剪贴板
     * <ul>
     * <li>设置成功时会自动弹出"已复制到剪贴板"</li>
     * <li>设置失败时会自动弹出具体错误信息</li>
     * </ul>
     *
     * @param text 要复制到剪贴板的文本内容
     * @return true表示设置成功，false表示设置失败（比如文本太大）
     */
    boolean setClipboardText(CharSequence text);

    /**
     * 将文本设置到剪贴板
     * <ul>
     * <li>设置成功时会自动弹出"successMessage"，除非successMessage为null</li>
     * <li>设置失败时会自动弹出具体错误信息</li>
     * </ul>
     *
     * @param text           要复制到剪贴板的文本内容
     * @param successMessage 复制成功时弹出的信息，为null则不弹出
     * @return true表示设置成功，false表示设置失败（比如文本太大）
     */
    boolean setClipboardText(CharSequence text, @Nullable String successMessage);

    /**
     * 显示短时间的Toast提示消息，可以在非UI线程调用
     * <p>
     * 如果msg类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param msg 要显示的消息内容
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    void showToast(CharSequence msg);

    /**
     * 显示短时间的Toast提示消息，并支持格式化参数，可以在非UI线程调用
     * <p>
     * 如果msg或formatArgs[n]类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param msg        要显示的消息内容
     * @param formatArgs 格式化参数，用于替换文本中的占位符（如%s、%d等）
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     * @see #getString(String, Object...)
     */
    void showToast(CharSequence msg, Object... formatArgs);

    /**
     * 显示长时间的Toast提示消息，可以在非UI线程调用
     * <p>
     * 如果msg或formatArgs[n]类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param msg 要显示的消息内容
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    void showToastL(CharSequence msg);

    /**
     * 显示长时间的Toast提示消息，并支持格式化参数，可以在非UI线程调用
     * <p>
     * 如果msg类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param msg        要显示的消息内容
     * @param formatArgs 格式化参数，用于替换文本中的占位符（如%s、%d等）
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     * @see #getString(String, Object...)
     */
    void showToastL(CharSequence msg, Object... formatArgs);

    /**
     * 取消当前正在显示的Toast提示消息
     */
    void cancelToast();

    /**
     * 写入日志消息
     * 日志可以在插件管理界面中查看，便于调试和问题排查
     *
     * @param msg 日志消息内容
     */
    void log(String msg);

    /**
     * 写入日志消息和异常信息
     *
     * @param msg 日志消息内容
     * @param e   异常对象，会记录异常的堆栈信息
     */
    void log(String msg, Throwable e);

    /**
     * 写入异常日志
     *
     * @param e 异常对象，会记录异常的堆栈信息
     */
    void log(Throwable e);

    /**
     * 打开插件日志查看器
     *
     * @see #log(String)
     * @see #log(Throwable)
     * @see #log(String, Throwable)
     */
    void openLogViewer();

}
