/*******************************************************************************
 * Copyright (c) 2013, 2015 EclipseSource.
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ******************************************************************************/
package bin.mt.json;

import androidx.annotation.NonNull;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import bin.mt.json.JSONObject.Member;
import bin.mt.plugin.api.ui.PluginCompoundButton;
import bin.mt.plugin.api.ui.PluginEditText;
import bin.mt.plugin.api.ui.PluginRadioGroup;
import bin.mt.plugin.api.ui.PluginSpinner;


/**
 * 表示一个 JSON 对象，即名称/值对的集合，其中名称是字符串，值是 JSON 值
 * <p>
 * 可以使用 <code>add(String, ...)</code> 方法添加成员，这些方法接受 {@link JSONValue} 实例、
 * 字符串、基本数字类型和布尔值。要修改对象的某些值，请使用 <code>put(String, ...)</code> 方法。
 * 请注意，<code>add</code> 方法比 <code>put</code> 更快，因为它们不会搜索现有成员。
 * 另一方面，<code>add</code> 方法不会阻止添加具有相同名称的多个成员。
 * JSON 不鼓励但不禁止重复名称
 * </p>
 * <p>
 * 可以通过名称使用 {@link #get(String)} 访问成员。可以从 {@link #names()} 方法获取所有名称的列表。
 * 此类还支持使用 {@link #iterator()} 或增强 for 循环按文档顺序遍历成员：
 * </p>
 * <pre>
 * for (Member member : jsonObject) {
 *   String name = member.getName();
 *   JSONValue value = member.getValue();
 *   ...
 * }
 * </pre>
 * <p>
 * 尽管 JSON 对象在定义上是无序的，但此类的实例保留成员的顺序，
 * 以便按文档顺序处理并保证可预测的输出
 * </p>
 * <p>
 * 注意，此类<strong>不是线程安全的</strong>。如果多个线程同时访问一个 <code>JSONObject</code>
 * 实例，且其中至少有一个线程修改了对象内容，则必须在外部同步对该实例的访问。
 * 否则可能导致不一致的状态
 * </p>
 * <p>
 * 此类<strong>不应被客户端继承</strong>
 * </p>
 */
public class JSONObject extends JSONValue implements Iterable<Member> {

    private final List<String> names;
    private final List<JSONValue> values;
    private transient HashIndexTable table;

    /**
     * 创建一个新的空 JSONObject
     */
    public JSONObject() {
        names = new ArrayList<>();
        values = new ArrayList<>();
        table = new HashIndexTable();
    }

    /**
     * 从给定的输入字符串解析创建一个新的 JSONObject
     */
    public JSONObject(String source) {
        this();
        if (source == null) {
            throw new NullPointerException("source is null");
        }
        JSON.DefaultHandler handler = new JSON.DefaultHandler();
        new JSONParser(handler).parseObject(this, source);
    }

    /**
     * 使用指定 JSON 对象的内容初始化创建一个新的 JSONObject
     *
     * @param object 要获取初始内容的 JSON 对象，不能为 <code>null</code>
     */
    public JSONObject(JSONObject object) {
        this(object, false);
    }

    private JSONObject(JSONObject object, boolean unmodifiable) {
        if (object == null) {
            throw new NullPointerException("object is null");
        }
        if (unmodifiable) {
            names = Collections.unmodifiableList(object.names);
            values = Collections.unmodifiableList(object.values);
        } else {
            names = new ArrayList<>(object.names);
            values = new ArrayList<>(object.values);
        }
        table = new HashIndexTable();
        updateHashIndex();
    }

    /**
     * 返回指定 JSONObject 的不可修改包装器。此方法允许提供对 JSONObject 的只读访问
     * <p>
     * 返回的 JSONObject 由给定对象支持并反映其变化。
     * 尝试修改返回的 JSONObject 将导致 <code>UnsupportedOperationException</code>
     * </p>
     *
     * @param object 要返回不可修改 JSONObject 的 JSONObject
     * @return 指定 JSONObject 的不可修改视图
     */
    public static JSONObject unmodifiableObject(JSONObject object) {
        return new JSONObject(object, true);
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定 <code>int</code> 值的 JSON 表示
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, int value) {
        add(name, JSON.value(value));
        return this;
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定 <code>long</code> 值的 JSON 表示
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, long value) {
        add(name, JSON.value(value));
        return this;
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定 <code>float</code> 值的 JSON 表示
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, float value) {
        add(name, JSON.value(value));
        return this;
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定 <code>double</code> 值的 JSON 表示
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, double value) {
        add(name, JSON.value(value));
        return this;
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定 <code>boolean</code> 值的 JSON 表示
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, boolean value) {
        add(name, JSON.value(value));
        return this;
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定字符串的 JSON 表示
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, String value) {
        add(name, JSON.value(value));
        return this;
    }

    /**
     * 将新成员追加到此对象末尾，具有指定的名称和指定的 JSON 值
     * <p>
     * 此方法<strong>不会阻止重复名称</strong>。使用已存在于对象中的名称调用此方法
     * 将追加另一个具有相同名称的成员。要替换现有成员，请改用 <code>put(name, value)</code> 方法。
     * 但是，<strong><em>add</em> 比 <em>put</em> 快得多</strong>（因为它不需要搜索现有成员）。
     * 因此在构造新对象时应优先使用 <em>add</em>
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值，不能为 <code>null</code>
     * @return 对象本身，以便链式调用
     */
    public JSONObject add(String name, JSONValue value) {
        if (name == null) {
            throw new NullPointerException("name is null");
        }
        if (value == null) {
            throw new NullPointerException("value is null");
        }
        table.add(name, names.size());
        names.add(name);
        values.add(value);
        return this;
    }

    /**
     * 将 {@link PluginEditText} 中的文本内容存储到此对象中。
     * 以编辑框的 ID 作为成员名称，编辑框的文本内容作为值
     *
     * @param editText 要获取文本的编辑框
     * @return 对象本身，以便链式调用
     */
    public JSONObject putText(PluginEditText editText) {
        put(editText.requireId(), editText.getText().toString());
        return this;
    }

    /**
     * 将 {@link PluginCompoundButton} 的选中状态存储到此对象中。
     * 以复合按钮的 ID 作为成员名称，选中状态作为布尔值
     *
     * @param compoundButton 要获取选中状态的复合按钮
     * @return 对象本身，以便链式调用
     */
    public JSONObject putChecked(PluginCompoundButton compoundButton) {
        put(compoundButton.requireId(), compoundButton.isChecked());
        return this;
    }

    /**
     * 将 {@link PluginSpinner} 的选中项索引存储到此对象中。
     * 以下拉框的 ID 作为成员名称，选中项的索引作为值
     *
     * @param spinner 要获取选中项的下拉框
     * @return 对象本身，以便链式调用
     */
    public JSONObject putSelection(PluginSpinner spinner) {
        put(spinner.requireId(), spinner.getSelection());
        return this;
    }

    /**
     * 将 {@link PluginRadioGroup} 中选中的单选按钮位置存储到此对象中。
     * 以单选组的 ID 作为成员名称，选中的单选按钮在组中的位置作为值
     *
     * @param radioGroup 要获取选中位置的单选组
     * @return 对象本身，以便链式调用
     */
    public JSONObject putCheckedPosition(PluginRadioGroup radioGroup) {
        put(radioGroup.getId(), radioGroup.getCheckedRadioButtonPosition());
        return this;
    }

    /**
     * 将 {@link PluginRadioGroup} 中选中的单选按钮 ID 存储到此对象中。
     * 以单选组的 ID 作为成员名称，选中的单选按钮的 ID 作为值
     *
     * @param radioGroup 要获取选中 ID 的单选组
     * @return 对象本身，以便链式调用
     */
    public JSONObject putCheckedId(PluginRadioGroup radioGroup) {
        put(radioGroup.getId(), radioGroup.getCheckedRadioButtonId());
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定 <code>int</code> 值的 JSON 表示。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要替换的成员的名称
     * @param value 要设置给成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, int value) {
        put(name, JSON.value(value));
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定 <code>long</code> 值的 JSON 表示。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要替换的成员的名称
     * @param value 要设置给成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, long value) {
        put(name, JSON.value(value));
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定 <code>float</code> 值的 JSON 表示。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, float value) {
        put(name, JSON.value(value));
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定 <code>double</code> 值的 JSON 表示。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, double value) {
        put(name, JSON.value(value));
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定 <code>boolean</code> 值的 JSON 表示。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, boolean value) {
        put(name, JSON.value(value));
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定字符串的 JSON 表示。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, String value) {
        put(name, JSON.value(value));
        return this;
    }

    /**
     * 将具有指定名称的成员的值设置为指定的 JSON 值。
     * 如果此对象不包含具有此名称的成员，则在对象末尾添加新成员。
     * 如果此对象包含多个具有此名称的成员，则只更改最后一个
     * <p>
     * 此方法<strong>应仅用于修改现有对象</strong>。要向新对象填充成员，
     * 应优先使用 <code>add(name, value)</code> 方法，它更快（因为不需要搜索现有成员）
     * </p>
     *
     * @param name  要添加的成员的名称
     * @param value 要添加的成员的值，不能为 <code>null</code>
     * @return 对象本身，以便链式调用
     */
    public JSONObject put(String name, JSONValue value) {
        if (name == null) {
            throw new NullPointerException("name is null");
        }
        if (value == null) {
            throw new NullPointerException("value is null");
        }
        int index = indexOf(name);
        if (index != -1) {
            values.set(index, value);
        } else {
            table.add(name, names.size());
            names.add(name);
            values.add(value);
        }
        return this;
    }

    /**
     * 从此对象中移除具有指定名称的成员。如果此对象包含多个具有给定名称的成员，
     * 则只移除最后一个。如果此对象不包含具有指定名称的成员，则不修改对象
     *
     * @param name 要移除的成员的名称
     * @return 对象本身，以便链式调用
     */
    public JSONObject remove(String name) {
        if (name == null) {
            throw new NullPointerException("name is null");
        }
        int index = indexOf(name);
        if (index != -1) {
            table.remove(index);
            names.remove(index);
            values.remove(index);
        }
        return this;
    }

    /**
     * 检查指定成员是否作为此对象的子元素存在。这不会测试此对象是否包含字面值 <code>null</code>，
     * 应使用 {@link JSONValue#isNull()} 来实现此目的
     *
     * @param name 要检查的成员的名称
     * @return 成员是否存在
     */
    public boolean contains(String name) {
        return names.contains(name);
    }

    /**
     * 将指定对象的所有成员复制到此对象中。当指定对象包含与此对象中已存在名称相同的成员时，
     * 此对象中的现有值将被指定对象中的对应值替换
     *
     * @param object 要合并的对象
     * @return 对象本身，以便链式调用
     */
    public JSONObject merge(JSONObject object) {
        if (object == null) {
            throw new NullPointerException("object is null");
        }
        for (Member member : object) {
            this.put(member.name, member.value);
        }
        return this;
    }

    /**
     * 返回此对象中具有指定名称的成员的值。如果此对象包含多个具有给定名称的成员，
     * 此方法将返回最后一个
     *
     * @param name 要返回其值的成员的名称
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回 <code>null</code>
     */
    public JSONValue get(String name) {
        if (name == null) {
            throw new NullPointerException("name is null");
        }
        int index = indexOf(name);
        return index != -1 ? values.get(index) : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link JSONObject} 值。
     * 如果此对象不包含具有此名称的成员，或者该成员的值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 JSONObject 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public JSONObject getJSONObject(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asObject() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link JSONArray} 值。
     * 如果此对象不包含具有此名称的成员，或者该成员的值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 JSONArray 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public JSONArray getJSONArray(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asArray() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>int</code> 值。如果此对象不包含具有此名称的成员，
     * 则返回给定的默认值。如果此对象包含多个具有给定名称的成员，则选择最后一个。
     * 如果该成员的值不表示 JSON 数字或无法解释为 Java <code>int</code>，则抛出异常
     *
     * @param name         要返回其值的成员的名称
     * @param defaultValue 如果请求的成员缺失时要返回的值
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回给定的默认值
     */
    public int getInt(String name, int defaultValue) {
        JSONValue value = get(name);
        return value != null ? value.asInt() : defaultValue;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>int</code> 值。
     * 等同于 <code>getInt(name, 0)</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 int 值，如果成员不存在则返回 0
     */
    public int getInt(String name) {
        return getInt(name, 0);
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link Integer} 值。
     * 如果成员不存在或值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 Integer 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public Integer getIntegerObject(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asInt() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>long</code> 值。如果此对象不包含具有此名称的成员，
     * 则返回给定的默认值。如果此对象包含多个具有给定名称的成员，则选择最后一个。
     * 如果该成员的值不表示 JSON 数字或无法解释为 Java <code>long</code>，则抛出异常
     *
     * @param name         要返回其值的成员的名称
     * @param defaultValue 如果请求的成员缺失时要返回的值
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回给定的默认值
     */
    public long getLong(String name, long defaultValue) {
        JSONValue value = get(name);
        return value != null ? value.asLong() : defaultValue;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>long</code> 值。
     * 等同于 <code>getLong(name, 0L)</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 long 值，如果成员不存在则返回 0L
     */
    public long getLong(String name) {
        return getLong(name, 0L);
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link Long} 值。
     * 如果成员不存在或值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 Long 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public Long getLongObject(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asLong() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>float</code> 值。如果此对象不包含具有此名称的成员，
     * 则返回给定的默认值。如果此对象包含多个具有给定名称的成员，则选择最后一个。
     * 如果该成员的值不表示 JSON 数字或无法解释为 Java <code>float</code>，则抛出异常
     *
     * @param name         要返回其值的成员的名称
     * @param defaultValue 如果请求的成员缺失时要返回的值
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回给定的默认值
     */
    public float getFloat(String name, float defaultValue) {
        JSONValue value = get(name);
        return value != null ? value.asFloat() : defaultValue;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>float</code> 值。
     * 等同于 <code>getFloat(name, 0F)</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 float 值，如果成员不存在则返回 0F
     */
    public float getFloat(String name) {
        return getFloat(name, 0F);
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link Float} 值。
     * 如果成员不存在或值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 Float 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public Float getFloatObject(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asFloat() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>double</code> 值。如果此对象不包含具有此名称的成员，
     * 则返回给定的默认值。如果此对象包含多个具有给定名称的成员，则选择最后一个。
     * 如果该成员的值不表示 JSON 数字或无法解释为 Java <code>double</code>，则抛出异常
     *
     * @param name         要返回其值的成员的名称
     * @param defaultValue 如果请求的成员缺失时要返回的值
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回给定的默认值
     */
    public double getDouble(String name, double defaultValue) {
        JSONValue value = get(name);
        return value != null ? value.asDouble() : defaultValue;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>double</code> 值。
     * 等同于 <code>getDouble(name, 0D)</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 double 值，如果成员不存在则返回 0D
     */
    public double getDouble(String name) {
        return getDouble(name, 0D);
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link Double} 值。
     * 如果成员不存在或值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 Double 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public Double getDoubleObject(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asDouble() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>boolean</code> 值。如果此对象不包含具有此名称的成员，
     * 则返回给定的默认值。如果此对象包含多个具有给定名称的成员，则选择最后一个。
     * 如果该成员的值不表示 JSON <code>true</code> 或 <code>false</code> 值，则抛出异常
     *
     * @param name         要返回其值的成员的名称
     * @param defaultValue 如果请求的成员缺失时要返回的值
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回给定的默认值
     */
    public boolean getBoolean(String name, boolean defaultValue) {
        JSONValue value = get(name);
        return value != null ? value.asBoolean() : defaultValue;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>boolean</code> 值。
     * 等同于 <code>getBoolean(name, false)</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 boolean 值，如果成员不存在则返回 false
     */
    public boolean getBoolean(String name) {
        return getBoolean(name, false);
    }

    /**
     * 返回此对象中具有指定名称的成员的 {@link Boolean} 值。
     * 如果成员不存在或值为 <code>null</code>，则返回 <code>null</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 Boolean 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public Boolean getBooleanObject(String name) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asBoolean() : null;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>String</code> 值。如果此对象不包含具有此名称的成员，
     * 则返回给定的默认值。如果此对象包含多个具有给定名称的成员，则选择最后一个。
     * 如果该成员的值不表示 JSON 字符串，则抛出异常
     *
     * @param name         要返回其值的成员的名称
     * @param defaultValue 如果请求的成员缺失时要返回的值
     * @return 具有指定名称的最后一个成员的值，如果此对象不包含具有该名称的成员则返回给定的默认值
     */
    public String getString(String name, String defaultValue) {
        JSONValue value = get(name);
        return value != null && !value.isNull() ? value.asString() : defaultValue;
    }

    /**
     * 返回此对象中具有指定名称的成员的 <code>String</code> 值。
     * 等同于 <code>getString(name, null)</code>
     *
     * @param name 要返回其值的成员的名称
     * @return 成员的 String 值，如果成员不存在或为 null 则返回 <code>null</code>
     */
    public String getString(String name) {
        return getString(name, null);
    }

    /**
     * 返回此对象中的成员（名称/值对）数量
     *
     * @return 此对象中的成员数量
     */
    public int size() {
        return names.size();
    }

    /**
     * 如果此对象不包含任何成员则返回 <code>true</code>
     *
     * @return 如果此对象不包含任何成员则返回 <code>true</code>
     */
    public boolean isEmpty() {
        return names.isEmpty();
    }

    /**
     * 按文档顺序返回此对象中名称的列表。返回的列表由此对象支持并将反映后续更改。
     * 它不能用于修改此对象。尝试修改返回的列表将导致异常
     *
     * @return 此对象中名称的列表
     */
    public List<String> names() {
        return Collections.unmodifiableList(names);
    }

    /**
     * 按文档顺序返回此对象中成员的迭代器。返回的迭代器不能用于修改此对象
     *
     * @return 此对象中成员的迭代器
     */
    @NonNull
    public Iterator<Member> iterator() {
        final Iterator<String> namesIterator = names.iterator();
        final Iterator<JSONValue> valuesIterator = values.iterator();
        return new Iterator<JSONObject.Member>() {

            public boolean hasNext() {
                return namesIterator.hasNext();
            }

            public Member next() {
                String name = namesIterator.next();
                JSONValue value = valuesIterator.next();
                return new Member(name, value);
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }

        };
    }

    @Override
    void write(JSONWriter writer) throws IOException {
        writer.writeObjectOpen();
        Iterator<String> namesIterator = names.iterator();
        Iterator<JSONValue> valuesIterator = values.iterator();
        if (namesIterator.hasNext()) {
            writer.writeMemberName(namesIterator.next());
            writer.writeMemberSeparator();
            valuesIterator.next().write(writer);
            while (namesIterator.hasNext()) {
                writer.writeObjectSeparator();
                writer.writeMemberName(namesIterator.next());
                writer.writeMemberSeparator();
                valuesIterator.next().write(writer);
            }
        }
        writer.writeObjectClose();
    }

    @Override
    public boolean isObject() {
        return true;
    }

    @Override
    public JSONObject asObject() {
        return this;
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = 31 * result + names.hashCode();
        result = 31 * result + values.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        JSONObject other = (JSONObject) obj;
        return names.equals(other.names) && values.equals(other.values);
    }

    int indexOf(String name) {
        int index = table.get(name);
        if (index != -1 && name.equals(names.get(index))) {
            return index;
        }
        return names.lastIndexOf(name);
    }

    private synchronized void readObject(ObjectInputStream inputStream)
            throws IOException, ClassNotFoundException {
        inputStream.defaultReadObject();
        table = new HashIndexTable();
        updateHashIndex();
    }

    private void updateHashIndex() {
        int size = names.size();
        for (int i = 0; i < size; i++) {
            table.add(names.get(i), i);
        }
    }

    /**
     * 表示 JSON 对象的成员，即名称和值的配对
     */
    public static class Member {

        private final String name;
        private final JSONValue value;

        Member(String name, JSONValue value) {
            this.name = name;
            this.value = value;
        }

        /**
         * 返回此成员的名称
         *
         * @return 此成员的名称，永不为 <code>null</code>
         */
        public String getName() {
            return name;
        }

        /**
         * 返回此成员的值
         *
         * @return 此成员的值，永不为 <code>null</code>
         */
        public JSONValue getValue() {
            return value;
        }

        @Override
        public int hashCode() {
            int result = 1;
            result = 31 * result + name.hashCode();
            result = 31 * result + value.hashCode();
            return result;
        }

        /**
         * 判断给定对象是否与此 JSONObject "相等"。如果对象也是 <code>JSONObject</code>
         * 且两个对象以<em>相同的顺序</em>包含相同的成员，则认为它们相等
         * <p>
         * 如果两个 JSONObject 相等，它们也将产生相同的 JSON 输出
         * </p>
         *
         * @param object 要与此 JSONObject 比较的对象
         * @return 如果指定对象与此 JSONObject 相等则返回 <tt>true</tt>，否则返回 <code>false</code>
         */
        @Override
        public boolean equals(Object object) {
            if (this == object) {
                return true;
            }
            if (object == null) {
                return false;
            }
            if (getClass() != object.getClass()) {
                return false;
            }
            Member other = (Member) object;
            return name.equals(other.name) && value.equals(other.value);
        }

    }

    static class HashIndexTable {

        private final byte[] hashTable = new byte[32]; // 必须是 2 的幂

        HashIndexTable() {
        }

        HashIndexTable(HashIndexTable original) {
            System.arraycopy(original.hashTable, 0, hashTable, 0, hashTable.length);
        }

        void add(String name, int index) {
            int slot = hashSlotFor(name);
            if (index < 0xff) {
                // 加 1，0 表示空
                hashTable[slot] = (byte) (index + 1);
            } else {
                hashTable[slot] = 0;
            }
        }

        void remove(int index) {
            for (int i = 0; i < hashTable.length; i++) {
                if ((hashTable[i] & 0xff) == index + 1) {
                    hashTable[i] = 0;
                } else if ((hashTable[i] & 0xff) > index + 1) {
                    hashTable[i]--;
                }
            }
        }

        int get(Object name) {
            int slot = hashSlotFor(name);
            // 减 1，0 表示空
            return (hashTable[slot] & 0xff) - 1;
        }

        private int hashSlotFor(Object element) {
            return element.hashCode() & hashTable.length - 1;
        }

    }

}
