package bin.mt.plugin.api.ui.menu;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * 插件弹出菜单接口
 * <p>
 * 提供弹出式菜单的核心功能，可在指定的锚点视图附近显示菜单，类似于Android原生的PopupMenu
 * <p>
 * 通过 {@code pluginUI.createPopupMenu(anchor)} 创建弹出菜单实例，其中anchor为锚点视图，
 * 菜单将相对于该视图显示。可以通过 {@link #setGravity(int)} 设置菜单的对齐方式。
 * <p>
 * 基本使用示例：
 * <pre>{@code
 * // 创建弹出菜单
 * PluginPopupMenu popupMenu = pluginUI.createPopupMenu(anchorView);
 *
 * // 获取菜单并添加菜单项
 * PluginMenu menu = popupMenu.getMenu();
 * menu.add("item1", "菜单项 1");
 * menu.add("item2", "菜单项 2");
 * menu.addSubMenu("submenu", "子菜单");
 *
 * // 设置菜单项点击监听器
 * popupMenu.setOnMenuItemClickListener(item -> {
 *     // 处理菜单项点击事件
 *     if ("item1".equals(item.getItemId())) {
 *         // 处理菜单项1的点击
 *     }
 *     return true; // 返回true表示已处理该事件
 * });
 *
 * // 显示菜单
 * popupMenu.show();
 * }</pre>
 */
public interface PluginPopupMenu {

    /**
     * 获取弹出菜单的菜单对象
     *
     * @return 菜单对象，用于添加菜单项
     */
    @NonNull
    PluginMenu getMenu();

    /**
     * 设置弹出菜单的对齐方式
     *
     * @param gravity 对齐方式，使用 Gravity 常量（如 Gravity.START, Gravity.END等）
     */
    void setGravity(int gravity);

    /**
     * 获取弹出菜单的对齐方式
     *
     * @return 对齐方式常量值
     */
    int getGravity();

    /**
     * 显示弹出菜单
     */
    void show();

    /**
     * 关闭弹出菜单
     */
    void dismiss();

    /**
     * 设置菜单项点击监听器
     *
     * @param listener 点击监听器，null表示移除监听器
     */
    void setOnMenuItemClickListener(@Nullable PluginMenu.OnMenuItemClickListener listener);

    /**
     * 设置菜单关闭监听器
     *
     * @param listener 关闭监听器，null表示移除监听器
     */
    void setOnDismissListener(@Nullable OnDismissListener listener);

    /**
     * 弹出菜单关闭监听器接口
     */
    interface OnDismissListener {
        /**
         * 弹出菜单被关闭时的回调
         *
         * @param menu 被关闭的弹出菜单
         */
        void onDismiss(PluginPopupMenu menu);
    }

}
