package bin.mt.plugin.api.ui.menu;

import android.content.res.ColorStateList;
import android.graphics.drawable.Drawable;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;

/**
 * 插件菜单项接口
 * <p>
 * 表示菜单中的一个单独项目，支持设置标题、图标、选中状态等属性
 */
public interface PluginMenuItem {

    /**
     * 获取菜单项的唯一标识符
     *
     * @return 菜单项ID
     */
    @NonNull
    String getItemId();

    /**
     * 获取菜单项所属的分组ID
     *
     * @return 分组ID，如果未分组则返回null
     */
    @Nullable
    String getGroupId();

    /**
     * 设置菜单项的标题
     * <p>
     * 如果title类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     *
     * @param title 菜单项标题
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setTitle(@NonNull CharSequence title);

    /**
     * 获取菜单项标题
     *
     * @return 菜单项的标题文本
     */
    @NonNull
    CharSequence getTitle();

    /**
     * 设置菜单项图标
     *
     * @param icon 图标Drawable，null表示移除图标
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setIcon(@Nullable Drawable icon);

    /**
     * 获取菜单项图标
     *
     * @return 图标Drawable，如果未设置则返回null
     */
    @Nullable
    Drawable getIcon();

    /**
     * 设置图标的着色列表
     *
     * @param tint 颜色状态列表，null表示移除着色
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setIconTintList(@Nullable ColorStateList tint);

    /**
     * 获取图标的着色列表
     *
     * @return 颜色状态列表，如果未设置则返回null
     */
    @Nullable
    ColorStateList getIconTintList();

    /**
     * 设置菜单项是否可选中
     *
     * @param checkable 是否可选中
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setCheckable(boolean checkable);

    /**
     * 获取菜单项是否可选中
     *
     * @return true表示可选中，false表示不可选中
     */
    boolean isCheckable();

    /**
     * 设置菜单项的选中状态
     *
     * @param checked 是否选中
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setChecked(boolean checked);

    /**
     * 获取菜单项是否被选中
     *
     * @return true表示已选中，false表示未选中
     */
    boolean isChecked();

    /**
     * 设置菜单项的可见性
     *
     * @param visible 是否可见
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setVisible(boolean visible);

    /**
     * 获取菜单项是否可见
     *
     * @return true表示可见，false表示不可见
     */
    boolean isVisible();

    /**
     * 设置菜单项的启用状态
     *
     * @param enabled 是否启用
     * @return 当前菜单项实例，支持链式调用
     */
    @NonNull
    PluginMenuItem setEnabled(boolean enabled);

    /**
     * 获取菜单项是否启用
     *
     * @return true表示启用，false表示禁用
     */
    boolean isEnabled();

    /**
     * 判断菜单项是否包含子菜单
     *
     * @return true表示包含子菜单，false表示不包含
     */
    boolean hasSubMenu();

    /**
     * 获取菜单项的子菜单
     *
     * @return 子菜单对象，如果没有子菜单则返回null
     */
    @Nullable
    PluginSubMenu getSubMenu();

}
