package bin.mt.plugin.api.ui.builder;

import android.graphics.drawable.Drawable;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.plugin.api.ui.PluginView;

/**
 * 插件视图构建器基础接口
 * <p>
 * 该接口是所有插件视图构建器的基础接口，提供了视图的通用属性设置功能。
 * 所有的插件视图构建器都继承自此接口，包括布局属性、尺寸设置、边距设置、
 * 事件监听器、可见性控制等基础功能。
 * </p>
 * <p>
 * 构建器模式采用链式调用设计，所有方法都返回构建器实例本身，
 * 支持流畅的方法链调用。使用泛型T确保链式调用的类型安全。
 * </p>
 *
 * @param <T> 具体的构建器类型，继承自PluginUIBuilder，用于支持链式调用的类型安全
 * @see PluginUIBuilder
 * @see PluginBaseTextViewBuilder
 * @see PluginBaseViewGroupBuilder
 * @see PluginEditTextBuilder
 * @see PluginRootLayoutBuilder
 * @see PluginSpinnerBuilder
 */
public interface PluginBaseViewBuilder<T extends PluginUIBuilder> extends PluginUIBuilder {

    /**
     * 设置视图的ID
     * <p>
     * ID用于在视图层次结构中标识一个视图组件，便于后续的查找和操作。
     * 设置ID后可以通过findViewById()方法进行查找。
     * </p>
     *
     * @param id 视图ID，不能为null或空字符串
     * @return 当前构建器实例，支持链式调用
     *
     * @throws NullPointerException 如果id为null或空字符串
     */
    T id(@NonNull String id);

    /**
     * 获取当前设置的视图ID
     *
     * @return 视图ID，如果未设置则返回null
     */
    @Nullable
    String getId();

    /**
     * 获取必需的视图ID
     * <p>
     * 与getId()不同，该方法要求视图必须具有ID，如果没有设置ID则会抛出异常。
     * 适用于必须有ID的场景。
     * </p>
     *
     * @return 视图ID，永远不为null
     *
     * @throws NullPointerException 如果视图没有设置ID
     */
    @NonNull
    String requireId();

    /**
     * 设置视图的标签对象
     * <p>
     * 标签可以是任意对象，用于存储与视图相关的自定义数据。
     * 这是一种灵活的方式来关联额外信息到视图上。
     * </p>
     *
     * @param tag 标签对象，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    T tag(Object tag);

    /**
     * 获取当前设置的标签对象
     *
     * @return 标签对象，可能为null
     */
    Object getTag();

    /**
     * 设置布局权重
     * <p>
     * 在支持权重的布局容器（如LinearLayout）中，权重决定了视图在剩余空间中的分配比例。
     * 权重值越大，分配到的空间越多。通常与宽度或高度设置为0dp配合使用。
     * </p>
     *
     * @param weight 权重值，通常为正数，0表示不参与权重分配
     * @return 当前构建器实例，支持链式调用
     */
    T layoutWeight(float weight);

    /**
     * 设置布局对齐方式
     * <p>
     * 该值决定了视图在父容器中的对齐方式，如居中、靠左、靠右等。
     * 具体效果取决于父容器的布局类型。
     * </p>
     *
     * @param gravity 布局对齐方式，使用Gravity常量，如Gravity.CENTER、Gravity.LEFT等
     * @return 当前构建器实例，支持链式调用
     *
     * @see android.view.Gravity
     */
    T layoutGravity(int gravity);

    /**
     * 设置视图尺寸
     * <p>
     * 同时设置视图的宽度和高度布局参数。
     * 可以使用具体的像素值、MATCH_PARENT或WRAP_CONTENT。
     * </p>
     *
     * @param width  宽度参数，可以是具体px值、MATCH_PARENT或WRAP_CONTENT
     * @param height 高度参数，可以是具体px值、MATCH_PARENT或WRAP_CONTENT
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#MATCH_PARENT
     * @see PluginView#WRAP_CONTENT
     */
    T size(int width, int height);

    /**
     * 设置视图宽度
     *
     * @param width 宽度参数，可以是具体px值、MATCH_PARENT或WRAP_CONTENT
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#MATCH_PARENT
     * @see PluginView#WRAP_CONTENT
     */
    T width(int width);

    /**
     * 设置视图高度
     *
     * @param height 高度参数，可以是具体px值、MATCH_PARENT或WRAP_CONTENT
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#MATCH_PARENT
     * @see PluginView#WRAP_CONTENT
     */
    T height(int height);


    /**
     * 设置视图宽度参数为WRAP_CONTENT
     *
     * @return 当前构建器实例，支持链式调用
     * @see PluginView#WRAP_CONTENT
     */
    T widthWrapContent();

    /**
     * 设置视图宽度参数为MATCH_PARENT
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#MATCH_PARENT
     */
    T widthMatchParent();

    /**
     * 设置视图高度参数为WRAP_CONTENT
     *
     * @return 当前构建器实例，支持链式调用
     * @see PluginView#WRAP_CONTENT
     */
    T heightWrapContent();

    /**
     * 设置视图高度参数为MATCH_PARENT
     *
     * @return 当前构建器实例，支持链式调用
     * @see PluginView#MATCH_PARENT
     */
    T heightMatchParent();

    /**
     * 设置内边距
     * <p>
     * 内边距是PluginView内容与边框之间的空间，影响PluginView内部内容的显示位置。
     * </p>
     *
     * @param left   左侧内边距px值
     * @param top    顶部内边距px值
     * @param right  右侧内边距px值
     * @param bottom 底部内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T padding(int left, int top, int right, int bottom);

    /**
     * 设置内边距
     *
     * @param padding 内边距px值，应用于左侧、顶部、右侧和底部
     */
    T padding(int padding);

    /**
     * 设置左侧内边距
     *
     * @param padding 左侧内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingLeft(int padding);

    /**
     * 设置顶部内边距
     *
     * @param padding 顶部内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingTop(int padding);

    /**
     * 设置右侧内边距
     *
     * @param padding 右侧内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingRight(int padding);

    /**
     * 设置底部内边距
     *
     * @param padding 底部内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingBottom(int padding);

    /**
     * 设置水平内边距
     * <p>
     * 同时设置左侧和右侧内边距为相同值。
     * </p>
     *
     * @param padding 水平内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingHorizontal(int padding);

    /**
     * 设置垂直内边距
     * <p>
     * 同时设置顶部和底部内边距为相同值。
     * </p>
     *
     * @param padding 垂直内边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingVertical(int padding);

    /**
     * 设置外边距
     * <p>
     * 外边距是视图与其他视图或父容器边界之间的空间。
     * 外边距不会影响视图本身的尺寸，但会影响视图在布局中的位置。
     * 外边距的实际效果取决于父容器的布局类型。
     * </p>
     *
     * @param left   左侧外边距px值
     * @param top    顶部外边距px值
     * @param right  右侧外边距px值
     * @param bottom 底部外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T margin(int left, int top, int right, int bottom);

    /**
     * 设置外边距
     *
     * @param margin 外边距px值，应用于左侧、顶部、右侧和底部
     * @return 当前构建器实例，支持链式调用
     */
    T margin(int margin);

    /**
     * 设置左侧外边距
     *
     * @param margin 左侧外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T marginLeft(int margin);

    /**
     * 设置顶部外边距
     *
     * @param margin 顶部外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T marginTop(int margin);

    /**
     * 设置右侧外边距
     *
     * @param margin 右侧外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T marginRight(int margin);

    /**
     * 设置底部外边距
     *
     * @param margin 底部外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T marginBottom(int margin);

    /**
     * 设置水平外边距
     * <p>
     * 同时设置左侧和右侧外边距为相同值。
     * </p>
     *
     * @param margin 水平外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T marginHorizontal(int margin);

    /**
     * 设置垂直外边距
     * <p>
     * 同时设置顶部和底部外边距为相同值。
     * </p>
     *
     * @param margin 垂直外边距px值
     * @return 当前构建器实例，支持链式调用
     */
    T marginVertical(int margin);

    // ========== 以下是使用dp单位的尺寸设置方法 ==========

    /**
     * 使用dp单位设置视图尺寸
     * <p>
     * dp（density-independent pixels）是Android中与密度无关的像素单位，
     * 能够在不同密度的屏幕上保持一致的视觉效果。
     * </p>
     *
     * @param widthDp  宽度dp值
     * @param heightDp 高度dp值
     * @return 当前构建器实例，支持链式调用
     */
    T sizeDp(float widthDp, float heightDp);

    /**
     * 使用dp单位设置视图宽度
     *
     * @param widthDp 宽度dp值
     * @return 当前构建器实例，支持链式调用
     */
    T widthDp(float widthDp);

    /**
     * 使用dp单位设置视图高度
     *
     * @param heightDp 高度dp值
     * @return 当前构建器实例，支持链式调用
     */
    T heightDp(float heightDp);

    /**
     * 使用dp单位设置内边距
     *
     * @param left   左侧内边距dp值
     * @param top    顶部内边距dp值
     * @param right  右侧内边距dp值
     * @param bottom 底部内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingDp(float left, float top, float right, float bottom);

    /**
     * 使用dp单位设置内边距
     *
     * @param padding 内边距dp值，应用于左侧、顶部、右侧和底部
     * @return 当前构建器实例，支持链式调用
     */
    T paddingDp(float padding);

    /**
     * 使用dp单位设置左侧内边距
     *
     * @param padding 左侧内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingLeftDp(float padding);

    /**
     * 使用dp单位设置顶部内边距
     *
     * @param padding 顶部内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingTopDp(float padding);

    /**
     * 使用dp单位设置右侧内边距
     *
     * @param padding 右侧内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingRightDp(float padding);

    /**
     * 使用dp单位设置底部内边距
     *
     * @param padding 底部内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingBottomDp(float padding);

    /**
     * 使用dp单位设置水平内边距
     *
     * @param padding 水平内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingHorizontalDp(float padding);

    /**
     * 使用dp单位设置垂直内边距
     *
     * @param padding 垂直内边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T paddingVerticalDp(float padding);

    /**
     * 使用dp单位设置外边距
     *
     * @param left   左侧外边距dp值
     * @param top    顶部外边距dp值
     * @param right  右侧外边距dp值
     * @param bottom 底部外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginDp(float left, float top, float right, float bottom);

    /**
     * 使用dp单位设置外边距
     *
     * @param margin 外边距dp值，应用于左侧、顶部、右侧和底部
     * @return 当前构建器实例，支持链式调用
     */
    T marginDp(float margin);

    /**
     * 使用dp单位设置左侧外边距
     *
     * @param margin 左侧外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginLeftDp(float margin);

    /**
     * 使用dp单位设置顶部外边距
     *
     * @param margin 顶部外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginTopDp(float margin);

    /**
     * 使用dp单位设置右侧外边距
     *
     * @param margin 右侧外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginRightDp(float margin);

    /**
     * 使用dp单位设置底部外边距
     *
     * @param margin 底部外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginBottomDp(float margin);

    /**
     * 使用dp单位设置水平外边距
     *
     * @param margin 水平外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginHorizontalDp(float margin);

    /**
     * 使用dp单位设置垂直外边距
     *
     * @param margin 垂直外边距dp值
     * @return 当前构建器实例，支持链式调用
     */
    T marginVerticalDp(float margin);

    // ========== 事件监听器设置方法 ==========

    /**
     * 设置点击事件监听器
     * <p>
     * 为视图设置点击事件的监听器，当用户点击视图时会触发回调。
     * 传入null可以移除已设置的监听器。
     * </p>
     *
     * @param listener 点击事件监听器，可以为null
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView.OnClickListener
     */
    T onClick(PluginView.OnClickListener listener);

    /**
     * 设置长按事件监听器
     * <p>
     * 为视图设置长按事件的监听器，当用户长按视图时会触发回调。
     * 传入null可以移除已设置的监听器。
     * </p>
     *
     * @param listener 长按事件监听器，可以为null
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView.OnLongClickListener
     */
    T onLongClick(PluginView.OnLongClickListener listener);

    // ========== 视图状态控制方法 ==========

    /**
     * 设置视图启用状态
     * <p>
     * 控制视图是否能响应用户交互。禁用视图后，
     * 视图将不响应点击、焦点等交互事件，并可能改变外观显示为灰色。
     * </p>
     *
     * @param enable true启用视图，false禁用视图
     * @return 当前构建器实例，支持链式调用
     */
    T enable(boolean enable);

    /**
     * 设置视图背景
     * <p>
     * 为视图设置背景图片或图形。背景会显示在视图内容的后面。
     * 传入null可以移除背景。
     * </p>
     *
     * @param background 背景Drawable对象，可以为null
     * @return 当前构建器实例，支持链式调用
     */
    T background(Drawable background);

    /**
     * 设置视图背景颜色
     * <p>
     * 为视图设置纯色背景。这是设置背景的便捷方法，
     * 内部会创建ColorDrawable对象。
     * </p>
     *
     * @param color 背景颜色值，格式为ARGB整数值
     * @return 当前构建器实例，支持链式调用
     */
    T backgroundColor(int color);

    /**
     * 设置视图透明度
     * <p>
     * 设置视图的透明度，影响视图及其所有子View的显示效果。
     * 0.0表示完全透明（不可见），1.0表示完全不透明（正常显示）。
     * </p>
     *
     * @param alpha 透明度值，范围为0.0-1.0
     * @return 当前构建器实例，支持链式调用
     */
    T alpha(float alpha);

    /**
     * 请求获取焦点
     *
     * @return 当前构建器实例，支持链式调用
     */
    T requestFocus();

    // ========== 可见性控制方法 ==========

    /**
     * 设置视图为可见状态
     * <p>
     * 等价于setVisibility(VISIBLE)，使视图可见并占用布局空间。
     * </p>
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#VISIBLE
     */
    T visible();

    /**
     * 设置视图为不可见状态
     * <p>
     * 等价于setVisibility(INVISIBLE)，使视图不可见但仍占用布局空间。
     * 视图在布局中的位置会被保留，但内容不显示。
     * </p>
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#INVISIBLE
     */
    T invisible();

    /**
     * 设置视图为消失状态
     * <p>
     * 等价于setVisibility(GONE)，使视图不可见且不占用布局空间。
     * 视图会完全从布局中移除，不影响其他视图的布局。
     * </p>
     *
     * @return 当前构建器实例，支持链式调用
     *
     * @see PluginView#GONE
     */
    T gone();

    /**
     * 统一指定子View的宽度，以最宽的View为准
     * <p>
     * 具体介绍接请看 {@link PluginView#unifyWidth(String...)} 方法的注释
     * </p>
     * <p>
     * <strong>使用示例：</strong>
     * <pre>{@code
     * pluginUIBuilder
     *     .addHorizontalLayout().children(builder -> builder
     *         .addTextView("label1").text("用户名")
     *         .addEditText().hint("请输入用户名")
     *     )
     *     .addHorizontalLayout().children(builder -> builder
     *         .addTextView("label2").text("电子邮箱地址")
     *         .addEditText().hint("请输入邮箱")
     *     )
     *     .unifyWidth("label1", "label2"); // 让两个标签宽度保持一致
     * }</pre>
     * 执行后，"用户名"和"电子邮箱地址"两个TextView将具有相同的宽度（以较长的"电子邮箱地址"为准），
     * 从而使得两行中的EditText能够左边对齐，界面更加整齐。
     * </p>
     *
     * @param ids 需要统一宽度的子View的id数组，至少需要传入2个id才有意义
     * @return 当前构建器实例，支持链式调用
     * @throws IllegalArgumentException 如果ids数组为空或长度小于2
     * @throws NullPointerException     如果某个id对应的View不存在
     */
    T unifyWidth(String... ids);

}
