package bin.mt.plugin.api.ui;

import android.graphics.drawable.Drawable;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * 插件视图基础接口
 */
public interface PluginView {

    /**
     * 大小参数：匹配父容器尺寸
     */
    int MATCH_PARENT = ViewGroup.LayoutParams.MATCH_PARENT;

    /**
     * 大小参数：根据内容自适应尺寸
     */
    int WRAP_CONTENT = ViewGroup.LayoutParams.WRAP_CONTENT;

    /**
     * 可见性：可见
     */
    int VISIBLE = View.VISIBLE;

    /**
     * 可见性：不可见但占用布局空间
     */
    int INVISIBLE = View.INVISIBLE;

    /**
     * 可见性：不可见且不占用布局空间
     */
    int GONE = View.GONE;

    /**
     * 获取绑定的PluginUI对象，每个PluginView都与一个PluginUI实例关联。
     *
     * @return 绑定的PluginUI对象，永远不为null
     */
    @NonNull
    PluginUI getPluginUI();

    /**
     * 获取PluginView的ID
     * <p>
     * PluginView的ID在对应的Builder类中通过id()方法设置，便于后续查找和操作。
     * </p>
     *
     * @return PluginView的ID字符串，没有设置ID，则返回null。
     */
    @Nullable
    String getId();

    /**
     * 获取PluginView的ID
     * <p>
     * 与getId()不同，该方法要求PluginView必须具有ID，如果没有ID则会抛出异常。
     * </p>
     *
     * @return PluginView的ID字符串，永远不为null
     *
     * @throws NullPointerException 如果PluginView没有设置ID
     */
    @NonNull
    String requireId();

    /**
     * 获取父视图容器
     * <p>
     * 返回当前PluginView的直接父容器，如果当前视图是根视图则返回null。
     * </p>
     *
     * @return 父视图容器，如果没有父容器则返回null
     */
    @Nullable
    PluginViewGroup getParentView();

    /**
     * 获取父视图容器
     * <p>
     * 与getParentView()不同，该方法要求PluginView必须具有父容器，如果没有父容器则会抛出异常。
     * </p>
     *
     * @return 父视图容器，永远不为null
     * @throws NullPointerException 如果PluginView没有父容器
     */
    @NonNull
    PluginViewGroup requireParentView();

    /**
     * 获取根视图容器
     * <p>
     * 返回当前PluginView所在视图树的根容器，即最顶层的PluginViewGroup。
     * 如果当前视图本身就是根视图，则返回自身的父容器。
     * </p>
     *
     * @return 根视图容器，永远不为null
     */
    @NonNull
    PluginViewGroup getRootView();

    /**
     * 获取PluginView的标签对象
     * <p>
     * 标签可以是任意对象，用于存储与PluginView相关的自定义数据。
     * 这是一种灵活的方式来关联额外信息到PluginView上。
     * </p>
     *
     * @return 标签对象，可能为null
     */
    Object getTag();

    /**
     * 设置PluginView的标签对象
     * <p>
     * 为PluginView设置一个标签对象，可以用于存储任意自定义数据。
     * </p>
     *
     * @param tag 要设置的标签对象，可以为null
     */
    void setTag(Object tag);

    /**
     * 获取布局权重
     * <p>
     * 在PluginLinearLayout等支持权重的布局中，权重决定了PluginView在剩余空间中的分配比例。
     * 权重值越大，分配到的空间越多。
     * </p>
     *
     * @return 当前的布局权重值
     */
    float getLayoutWeight();

    /**
     * 设置布局权重
     * <p>
     * 设置PluginView在父布局中的权重，只在支持权重的布局容器中生效。
     * 通常与宽度或高度设置为0dp配合使用。
     * </p>
     *
     * @param weight 权重值，通常为正数
     */
    void setLayoutWeight(float weight);

    /**
     * 获取布局对齐方式
     * <p>
     * 该值决定了PluginView在父容器中的对齐方式，如居中、靠左、靠右等。
     * 具体效果取决于父容器的布局类型。
     * </p>
     *
     * @return 当前的布局对齐方式
     *
     * @see android.view.Gravity
     */
    int getLayoutGravity();

    /**
     * 设置布局对齐方式
     * <p>
     * 设置PluginView在父容器中的对齐方式。常用值包括：
     * Gravity.CENTER、Gravity.LEFT、Gravity.RIGHT等。
     * </p>
     *
     * @param gravity 布局对齐方式，使用Gravity常量
     * @see android.view.Gravity
     */
    void setLayoutGravity(int gravity);

    /**
     * 获取宽度参数
     * <p>
     * 返回PluginView的宽度布局参数，可能是具体px值、WRAP_CONTENT或MATCH_PARENT。
     * 注意：这不是PluginView的实际渲染宽度，如需获取实际宽度请使用getViewWidth()。
     * </p>
     *
     * @return 宽度参数值
     */
    int getWidth();

    /**
     * 获取高度参数
     * <p>
     * 返回PluginView的高度布局参数，可能是具体px值、WRAP_CONTENT或MATCH_PARENT。
     * 注意：这不是PluginView的实际渲染高度，如需获取实际高度请使用getViewHeight()。
     * </p>
     *
     * @return 高度参数值
     */
    int getHeight();

    /**
     * 获取PluginView当前真实宽度
     * <p>
     * 获取PluginView在屏幕上实际渲染的宽度（px）。
     * 只有当PluginView完成布局后才能获取到正确的值，否则返回0。
     * </p>
     *
     * @return PluginView实际宽度（px），未布局时返回0
     */
    int getViewWidth();

    /**
     * 获取PluginView当前真实高度
     * <p>
     * 获取PluginView在屏幕上实际渲染的高度（px）。
     * 只有当PluginView完成布局后才能获取到正确的值，否则返回0。
     * </p>
     *
     * @return PluginView实际高度（px），未布局时返回0
     */
    int getViewHeight();

    /**
     * 获取左侧内边距
     *
     * @return 左侧内边距px值
     */
    int getPaddingLeft();

    /**
     * 获取顶部内边距
     *
     * @return 顶部内边距px值
     */
    int getPaddingTop();

    /**
     * 获取右侧内边距
     *
     * @return 右侧内边距px值
     */
    int getPaddingRight();

    /**
     * 获取底部内边距
     *
     * @return 底部内边距px值
     */
    int getPaddingBottom();

    /**
     * 获取左侧外边距
     *
     * @return 左侧外边距px值
     */
    int getMarginLeft();

    /**
     * 获取顶部外边距
     *
     * @return 顶部外边距px值
     */
    int getMarginTop();

    /**
     * 获取右侧外边距
     *
     * @return 右侧外边距px值
     */
    int getMarginRight();

    /**
     * 获取底部外边距
     *
     * @return 底部外边距px值
     */
    int getMarginBottom();

    /**
     * 设置PluginView尺寸
     * <p>
     * 同时设置PluginView的宽度和高度布局参数。
     * 可以使用MATCH_PARENT、WRAP_CONTENT或具体px。
     * </p>
     *
     * @param width  宽度参数（px值、MATCH_PARENT或WRAP_CONTENT）
     * @param height 高度参数（px值、MATCH_PARENT或WRAP_CONTENT）
     */
    void setSize(int width, int height);

    /**
     * 设置PluginView宽度参数
     *
     * @param width 宽度参数（px值、MATCH_PARENT或WRAP_CONTENT）
     */
    void setWidth(int width);

    /**
     * 设置PluginView高度参数
     *
     * @param height 高度参数（px、MATCH_PARENT或WRAP_CONTENT）
     */
    void setHeight(int height);

    /**
     * 设置内边距
     * <p>
     * 内边距是PluginView内容与边框之间的空间，影响PluginView内部内容的显示位置。
     * </p>
     *
     * @param left   左侧内边距px值
     * @param top    顶部内边距px值
     * @param right  右侧内边距px值
     * @param bottom 底部内边距px值
     */
    void setPadding(int left, int top, int right, int bottom);

    /**
     * 设置内边距
     *
     * @param padding 内边距px值，应用于左侧、顶部、右侧和底部
     * @see #setPadding(int, int, int, int)
     */
    void setPadding(int padding);

    /**
     * 设置左侧内边距
     *
     * @param padding 左侧内边距px值
     * @see #setPadding(int, int, int, int)
     */
    void setPaddingLeft(int padding);

    /**
     * 设置顶部内边距
     *
     * @param padding 顶部内边距px值
     * @see #setPadding(int, int, int, int)
     */
    void setPaddingTop(int padding);

    /**
     * 设置右侧内边距
     *
     * @param padding 右侧内边距px值
     * @see #setPadding(int, int, int, int)
     */
    void setPaddingRight(int padding);

    /**
     * 设置底部内边距
     *
     * @param padding 底部内边距px值
     * @see #setPadding(int, int, int, int)
     */
    void setPaddingBottom(int padding);

    /**
     * 设置水平内边距
     * <p>
     * 同时设置左侧和右侧内边距为相同值。
     * </p>
     *
     * @param padding 水平内边距px值
     * @see #setPadding(int, int, int, int)
     */
    void setPaddingHorizontal(int padding);

    /**
     * 设置垂直内边距
     * <p>
     * 同时设置顶部和底部内边距为相同值。
     * </p>
     *
     * @param padding 垂直内边距px值
     * @see #setPadding(int, int, int, int)
     */
    void setPaddingVertical(int padding);

    /**
     * 设置外边距
     * <p>
     * 外边距是视图与其他视图或父容器边界之间的空间。
     * 外边距不会影响视图本身的尺寸，但会影响视图在布局中的位置。
     * 外边距的实际效果取决于父容器的布局类型。
     * </p>
     *
     * @param left   左侧外边距px值
     * @param top    顶部外边距px值
     * @param right  右侧外边距px值
     * @param bottom 底部外边距px值
     */
    void setMargin(int left, int top, int right, int bottom);

    /**
     * 设置外边距
     *
     * @param margin 外边距px值，应用于左侧、顶部、右侧和底部
     * @see #setMargin(int, int, int, int)
     */
    void setMargin(int margin);

    /**
     * 设置左侧外边距
     *
     * @param margin 左侧外边距px值
     * @see #setMargin(int, int, int, int)
     */
    void setMarginLeft(int margin);

    /**
     * 设置顶部外边距
     *
     * @param margin 顶部外边距px值
     * @see #setMargin(int, int, int, int)
     */
    void setMarginTop(int margin);

    /**
     * 设置右侧外边距
     *
     * @param margin 右侧外边距px值
     * @see #setMargin(int, int, int, int)
     */
    void setMarginRight(int margin);

    /**
     * 设置底部外边距
     *
     * @param margin 底部外边距px值
     * @see #setMargin(int, int, int, int)
     */
    void setMarginBottom(int margin);

    /**
     * 设置水平外边距
     * <p>
     * 同时设置左侧和右侧外边距为相同值。
     * </p>
     *
     * @param margin 水平外边距px值
     * @see #setMargin(int, int, int, int)
     */
    void setMarginHorizontal(int margin);

    /**
     * 设置垂直外边距
     * <p>
     * 同时设置顶部和底部外边距为相同值。
     * </p>
     *
     * @param margin 垂直外边距px值
     * @see #setMargin(int, int, int, int)
     */
    void setMarginVertical(int margin);

    // dp单位的尺寸设置方法

    /**
     * 使用dp单位设置PluginView尺寸
     *
     * @param widthDp  宽度dp值
     * @param heightDp 高度dp值
     */
    void setSizeDp(float widthDp, float heightDp);

    /**
     * 使用dp单位设置PluginView宽度
     *
     * @param widthDp 宽度dp值
     */
    void setWidthDp(float widthDp);

    /**
     * 使用dp单位设置PluginView高度
     *
     * @param heightDp 高度dp值
     */
    void setHeightDp(float heightDp);

    /**
     * 使用dp单位设置内边距
     *
     * @param left   左侧内边距dp值
     * @param top    顶部内边距dp值
     * @param right  右侧内边距dp值
     * @param bottom 底部内边距dp值
     */
    void setPaddingDp(float left, float top, float right, float bottom);

    /**
     * 使用dp单位设置内边距
     *
     * @param padding 内边距dp值，应用于左侧、顶部、右侧和底部
     */
    void setPaddingDp(float padding);

    /**
     * 使用dp单位设置左侧内边距
     *
     * @param padding 左侧内边距dp值
     */
    void setPaddingLeftDp(float padding);

    /**
     * 使用dp单位设置顶部内边距
     *
     * @param padding 顶部内边距dp值
     */
    void setPaddingTopDp(float padding);

    /**
     * 使用dp单位设置右侧内边距
     *
     * @param padding 右侧内边距dp值
     */
    void setPaddingRightDp(float padding);

    /**
     * 使用dp单位设置底部内边距
     *
     * @param padding 底部内边距dp值
     */
    void setPaddingBottomDp(float padding);

    /**
     * 使用dp单位设置水平内边距
     *
     * @param padding 水平内边距dp值
     */
    void setPaddingHorizontalDp(float padding);

    /**
     * 使用dp单位设置垂直内边距
     *
     * @param padding 垂直内边距dp值
     */
    void setPaddingVerticalDp(float padding);

    /**
     * 使用dp单位设置外边距
     *
     * @param left   左侧外边距dp值
     * @param top    顶部外边距dp值
     * @param right  右侧外边距dp值
     * @param bottom 底部外边距dp值
     */
    void setMarginDp(float left, float top, float right, float bottom);

    /**
     * 使用dp单位设置外边距
     *
     * @param margin 外边距dp值，应用于左侧、顶部、右侧和底部
     */
    void setMarginDp(float margin);

    /**
     * 使用dp单位设置左侧外边距
     *
     * @param margin 左侧外边距dp值
     */
    void setMarginLeftDp(float margin);

    /**
     * 使用dp单位设置顶部外边距
     *
     * @param margin 顶部外边距dp值
     */
    void setMarginTopDp(float margin);

    /**
     * 使用dp单位设置右侧外边距
     *
     * @param margin 右侧外边距dp值
     */
    void setMarginRightDp(float margin);

    /**
     * 使用dp单位设置底部外边距
     *
     * @param margin 底部外边距dp值
     */
    void setMarginBottomDp(float margin);

    /**
     * 使用dp单位设置水平外边距
     *
     * @param margin 水平外边距dp值
     */
    void setMarginHorizontalDp(float margin);

    /**
     * 使用dp单位设置垂直外边距
     *
     * @param margin 垂直外边距dp值
     */
    void setMarginVerticalDp(float margin);

    /**
     * 设置点击事件监听器
     * <p>
     * 为PluginView设置点击事件的监听器，当用户点击PluginView时会触发回调。
     * 传入null可以移除已设置的监听器。
     * </p>
     *
     * @param listener 点击事件监听器，可以为null
     * @see OnClickListener
     */
    void setOnClickListener(@Nullable PluginView.OnClickListener listener);

    /**
     * 设置长按事件监听器
     * <p>
     * 为PluginView设置长按事件的监听器，当用户长按PluginView时会触发回调。
     * 传入null可以移除已设置的监听器。
     * </p>
     *
     * @param listener 长按事件监听器，可以为null
     * @see OnLongClickListener
     */
    void setOnLongClickListener(@Nullable PluginView.OnLongClickListener listener);

    /**
     * 检查PluginView是否启用
     * <p>
     * 启用状态决定了PluginView是否能响应用户交互。
     * 禁用的PluginView通常会显示为灰色且不响应点击等事件。
     * </p>
     *
     * @return true表示PluginView已启用，false表示已禁用
     */
    boolean isEnabled();

    /**
     * 设置PluginView启用状态
     * <p>
     * 控制PluginView是否能响应用户交互。禁用PluginView后，
     * PluginView将不响应点击、焦点等交互事件，并可能改变外观。
     * </p>
     *
     * @param enabled true启用PluginView，false禁用PluginView
     */
    void setEnabled(boolean enabled);

    /**
     * 获取PluginView背景
     * <p>
     * 返回当前设置的背景Drawable对象，如果通过 {@link #setBackgroundColor(int)} 设置背景颜色，此方法将
     * 返回一个 {@link android.graphics.drawable.ColorDrawable}；如果没有设置背景，可能返回null。
     * </p>
     *
     * @return 背景Drawable对象，可能为null
     */
    Drawable getBackground();

    /**
     * 设置PluginView背景
     * <p>
     * 为PluginView设置背景图片或图形。背景会显示在PluginView内容的后面。
     * 传入null可以移除背景。
     * </p>
     *
     * @param background 背景Drawable对象，可以为null
     */
    void setBackground(Drawable background);

    /**
     * 设置PluginView背景颜色
     * <p>
     * 为PluginView设置纯色背景。这是设置背景的便捷方法，
     * 内部会创建 {@link android.graphics.drawable.ColorDrawable} 对象。
     * </p>
     *
     * @param color 背景颜色值（ARGB格式）
     */
    void setBackgroundColor(int color);

    /**
     * 获取PluginView透明度
     * <p>
     * 返回PluginView当前的透明度值，范围从0.0（完全透明）到1.0（完全不透明）。
     * </p>
     *
     * @return 透明度值（0.0-1.0）
     */
    float getAlpha();

    /**
     * 设置PluginView透明度
     * <p>
     * 设置PluginView的透明度，影响PluginView及其所有子View的显示效果。
     * 0.0表示完全透明（不可见），1.0表示完全不透明（正常显示）。
     * </p>
     *
     * @param alpha 透明度值（0.0-1.0）
     */
    void setAlpha(float alpha);

    /**
     * 获取PluginView可见性状态
     * <p>
     * 返回PluginView当前的可见性状态，可能的值有：
     * VISIBLE、INVISIBLE、GONE
     * </p>
     *
     * @return 可见性状态值
     *
     * @see #VISIBLE
     * @see #INVISIBLE
     * @see #GONE
     */
    int getVisibility();

    /**
     * 设置PluginView为可见状态
     * <p>
     * 等价于setVisibility(VISIBLE)，使PluginView可见并占用布局空间。
     * </p>
     *
     * @see #setVisibility(int)
     * @see #VISIBLE
     */
    void setVisible();

    /**
     * 设置PluginView为不可见状态
     * <p>
     * 等价于setVisibility(INVISIBLE)，使PluginView不可见但仍占用布局空间。
     * </p>
     *
     * @see #setVisibility(int)
     * @see #INVISIBLE
     */
    void setInvisible();

    /**
     * 设置PluginView为消失状态
     * <p>
     * 等价于setVisibility(GONE)，使PluginView不可见且不占用布局空间。
     * </p>
     *
     * @see #setVisibility(int)
     * @see #GONE
     */
    void setGone();

    /**
     * 设置PluginView可见性
     * <p>
     * 控制PluginView的可见性状态：
     * - VISIBLE: 可见并占用空间
     * - INVISIBLE: 不可见但占用空间
     * - GONE: 不可见且不占用空间
     * </p>
     *
     * @param visibility 可见性状态（VISIBLE、INVISIBLE或GONE）
     * @see #VISIBLE
     * @see #INVISIBLE
     * @see #GONE
     */
    void setVisibility(int visibility);

    /**
     * 判断当前PluginView是否拥有焦点
     *
     * @return 是否拥有焦点
     */
    boolean isFocused();

    /**
     * 请求获取焦点
     *
     * @return 是否成功获取焦点
     */
    boolean requestFocus();

    /**
     * 根据ID查找子View
     * <p>
     * 在当前PluginView及其子View层次结构中查找指定ID的PluginView。
     * </p>
     *
     * @param id  要查找的PluginViewID
     * @param <T> 期望的PluginView类型
     * @return 找到的PluginView对象，未找到时返回null
     */
    @Nullable
    <T extends PluginView> T findViewById(String id);

    /**
     * 根据ID查找必需的子View
     * <p>
     * 与findViewById()类似，但要求必须找到指定的PluginView，
     * 如果未找到会抛出异常。适用于必须存在的PluginView查找场景。
     * </p>
     *
     * @param id  要查找的PluginViewID
     * @param <T> 期望的PluginView类型
     * @return 找到的PluginView对象，永远不为null
     *
     * @throws NullPointerException 如果未找到指定ID的PluginView
     */
    @NonNull
    <T extends PluginView> T requireViewById(String id);

    /**
     * 统一指定子View的宽度，以最宽的View为准
     * <p>
     * 该方法会通过requireViewById()将传入的id转为对应的View，测量它们的实际宽度，
     * 然后将所有这些View的宽度设置为其中最宽View的宽度，从而实现宽度对齐效果。
     * </p>
     *
     * <p>
     * <strong>使用场景：</strong><br>
     * 常用于表单布局中，让标签文本保持相同宽度，使得后面的输入框能够右边对齐，
     * 提升界面的整齐美观度。
     * </p>
     *
     * <p>
     * <strong>使用示例：</strong>
     * <pre>{@code
     * PluginView view = pluginUIBuilder
     *     .addHorizontalLayout().children(builder -> builder
     *         .addTextView("label1").text("用户名")
     *         .addEditText().hint("请输入用户名")
     *     )
     *     .addHorizontalLayout().children(builder -> builder
     *         .addTextView("label2").text("电子邮箱地址")
     *         .addEditText().hint("请输入邮箱")
     *     )
     *     .build();
     * view.unifyWidth("label1", "label2"); // 让两个标签宽度保持一致
     * }</pre>
     * 执行后，"用户名"和"电子邮箱地址"两个TextView将具有相同的宽度（以较长的"电子邮箱地址"为准），
     * 从而使得两行中的EditText能够左边对齐，界面更加整齐。
     * </p>
     *
     * <p>
     * <strong>重要提示：</strong>
     * <ul>
     * <li>参与对齐的View建议使用WRAP_CONTENT作为宽度参数，以获得最佳效果</li>
     * <li>如果View使用MATCH_PARENT或固定宽度，可能无法达到预期的对齐效果</li>
     * <li>传入的id必须是当前ViewGroup内的直接或间接子View的id</li>
     * </ul>
     * </p>
     *
     * @param ids 需要统一宽度的子View的id数组，至少需要传入2个id才有意义
     * @throws IllegalArgumentException 如果ids数组为空或长度小于2
     * @throws NullPointerException     如果某个id对应的View不存在
     */
    void unifyWidth(String... ids);

    /**
     * 统一指定子View的宽度，以最宽的View为准
     *
     * @param views 需要统一宽度的子View数组，至少需要传入2个View才有意义
     * @throws IllegalArgumentException 如果views数组为空或长度小于2
     * @see #unifyWidth(String...)
     */
    void unifyWidth(PluginView... views);

    /**
     * 点击事件监听器接口
     * <p>
     * 用于处理PluginView的点击事件，当用户点击PluginView时会调用onClick方法。
     * </p>
     */
    interface OnClickListener {

        /**
         * PluginView被点击时的回调方法
         *
         * @param view 被点击的PluginView对象
         */
        void onClick(@NonNull PluginView view);
    }

    /**
     * 长按事件监听器接口
     * <p>
     * 用于处理PluginView的长按事件，当用户长按PluginView时会调用onLongClick方法。
     * </p>
     */
    interface OnLongClickListener {

        /**
         * PluginView被长按时的回调方法
         *
         * @param view 被长按的PluginView对象
         * @return true表示消费了长按事件（会有震动效果），false表示不消费
         */
        boolean onLongClick(@NonNull PluginView view);
    }
}
