package bin.mt.plugin.api.translation;

import androidx.annotation.NonNull;

import bin.mt.plugin.api.PluginContext;

/**
 * 翻译引擎基础抽象类
 * 提供翻译引擎的基础实现和默认行为
 * 开发者可以继承此类来快速实现自定义翻译引擎
 */
public abstract class BaseTranslationEngine implements TranslationEngine {
    /**
     * 翻译引擎配置对象
     */
    private final Configuration configuration;

    /**
     * MT插件上下文
     */
    private PluginContext context;

    /**
     * 使用默认配置构造翻译引擎
     * 默认启用自动修复格式控制错误功能
     */
    public BaseTranslationEngine() {
        this.configuration = new ConfigurationBuilder()
                .setAutoRepairFormatSpecifiersError(true)  // 启用自动修复格式控制错误
                .build();
    }

    /**
     * 使用自定义配置构造翻译引擎
     *
     * @param configuration 自定义的翻译引擎配置
     */
    public BaseTranslationEngine(Configuration configuration) {
        this.configuration = configuration;
    }

    /**
     * 初始化翻译引擎（最终实现）
     * 此方法被标记为final，子类不能重写
     * 会自动保存context并调用子类的init()方法
     *
     * @param context MT插件上下文对象
     */
    @Override
    public final void init(PluginContext context) {
        this.context = context;
        init();  // 调用子类可重写的init方法
    }

    /**
     * 子类可重写的初始化方法
     * 在context设置完成后被调用
     * 子类可以在此方法中进行具体的初始化操作
     */
    protected void init() {
        // 默认空实现，子类可以重写
    }

    /**
     * 获取MT插件上下文对象
     *
     * @return 在init方法中传入的context对象
     */
    @Override
    public PluginContext getContext() {
        return context;
    }

    /**
     * 获取翻译引擎配置信息
     *
     * @return 翻译引擎的配置对象，不会返回null
     */
    @NonNull
    @Override
    public Configuration getConfiguration() {
        return configuration;
    }

    /**
     * 获取语言的显示名称
     * 使用MT内置的语言名称本地化功能
     *
     * @param language 语言代码，例如"en"、"zh-CN"等
     * @return 语言的本地化显示名称
     */
    @NonNull
    @Override
    public String getLanguageDisplayName(String language) {
        String name = getContext().getStringNullable("lang:" + language);
        return name != null ? name : language;
    }

    /**
     * 翻译开始前的准备操作（UI线程）
     * 默认空实现，子类可以重写此方法进行自定义操作
     * 此方法在UI线程中执行
     */
    @Override
    public void beforeStart() {
        // 默认空实现
    }

    /**
     * 开始翻译时的操作（子线程）
     * 默认空实现，子类可以重写此方法进行自定义操作
     * 此方法在子线程中执行
     */
    @Override
    public void onStart() {
        // 默认空实现
    }

    /**
     * 翻译结束时的操作（子线程）
     * 默认空实现，子类可以重写此方法进行自定义操作
     * 此方法在子线程中执行
     */
    @Override
    public void onFinish() {
        // 默认空实现
    }

    /**
     * 翻译结束后的操作（UI线程）
     * 默认空实现，子类可以重写此方法进行自定义操作
     * 此方法在UI线程中执行
     */
    @Override
    public void afterFinish() {
        // 默认空实现
    }

    /**
     * 翻译错误处理（UI线程）
     * 默认实现：将异常记录到日志并返回false
     * 子类可以重写此方法进行自定义错误处理
     *
     * @param e 发生的异常
     * @return true表示已处理错误，MT不会显示错误对话框；
     * false表示让MT显示默认错误对话框
     */
    @Override
    public boolean onError(Exception e) {
        getContext().log(e);  // 记录异常到日志
        return false;  // 让MT显示默认错误对话框
    }
}
