package bin.mt.plugin.api.regex;

/**
 * 正则表达式匹配结果快照
 */
public interface MatcherSnapshot {

    /**
     * 获取快照关联的文本序列
     *
     * @return 匹配时的字符序列
     */
    CharSequence getText();

    /**
     * 获取匹配记录中匹配子序列的起始位置
     *
     * @return 匹配子序列的起始索引
     */
    int start();

    /**
     * 获取匹配记录中指定捕获组的起始位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的起始索引
     */
    int start(int group);

    /**
     * 获取匹配记录中匹配子序列的结束位置
     *
     * @return 匹配子序列的结束索引（不包含该位置的字符）
     */
    int end();

    /**
     * 获取匹配记录中指定捕获组的结束位置
     *
     * @param group 捕获组的索引（0表示整个匹配）
     * @return 指定捕获组的结束索引（不包含该位置的字符）
     */
    int end(int group);

    /**
     * 获取匹配记录中匹配的子序列内容
     *
     * @return 匹配的字符串内容
     */
    String group();

    /**
     * 获取匹配记录中指定捕获组的内容
     *
     * @param group 捕获组的索引（0表示整个匹配，1及以上表示各个捕获组）
     * @return 指定捕获组的字符串内容
     */
    String group(int group);

    /**
     * 获取正则表达式中捕获组的数量
     *
     * @return 捕获组的数量（不包括组0，即整个匹配）
     */
    int groupCount();

    /**
     * 根据替换模式获取替换后的文本
     * <p>
     * 基于快照中的匹配结果进行文本替换
     *
     * @param replacementTemplate 替换字符串，可包含捕获组引用
     * @return 替换后的文本内容
     */
    String getReplacedText(String replacementTemplate);

    /**
     * 将替换结果追加到StringBuilder中
     * <p>
     * 基于快照中的匹配结果执行替换并追加到指定StringBuilder
     *
     * @param sb                  目标StringBuilder对象
     * @param replacementTemplate 替换模式字符串，可包含捕获组引用
     */
    void appendReplacement(StringBuilder sb, String replacementTemplate);

    /**
     * 预处理替换模式字符串
     * <p>
     * 解析替换模式中的捕获组引用，为后续替换操作做准备
     *
     * @param replacementTemplate 替换模式字符串
     */
    void prepareReplacement(String replacementTemplate);

    /**
     * 获取经过预处理计算后的替换结果
     * <p>
     * 需要先调用prepareReplacement()方法
     *
     * @return 计算后的替换文本
     * @throws IllegalStateException 未调用prepareReplacement()进行预处理
     */
    String getComputedReplacement();

    /**
     * 清除已计算的替换结果缓存
     * <p>
     * 释放prepareReplacement()产生的计算结果
     */
    void clearComputedReplacement();

}
