package bin.mt.plugin.api.editor;

import android.text.SpannableString;
import android.text.Spanned;
import android.text.style.ForegroundColorSpan;
import android.text.style.RelativeSizeSpan;

import androidx.annotation.NonNull;

import java.util.Objects;

import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.ui.PluginUI;

/**
 * 文本编辑器菜单的抽象基类
 */
public abstract class BaseTextEditorBaseMenu implements TextEditorBaseMenu {
    /**
     * 插件上下文实例
     * <p>
     * 在init()方法中由框架自动设置，用于访问插件运行环境提供的各种服务和资源。
     * 子类可通过getContext()方法获取此实例。
     */
    private PluginContext context;

    /**
     * 初始化方法的最终实现
     * <p>
     * 此方法被标记为final，确保上下文设置逻辑不会被子类意外覆盖。
     * 方法执行流程：
     * <ol>
     * <li>保存传入的插件上下文实例</li>
     * <li>调用无参的init()钩子方法供子类进行自定义初始化</li>
     * </ol>
     *
     * @param context MT插件上下文，提供插件运行所需的环境和资源访问能力
     */
    @Override
    public final void init(PluginContext context) {
        this.context = context;
        init();
    }

    /**
     * 子类自定义初始化钩子方法
     * <p>
     * 此方法在插件上下文设置完成后被调用，子类可以重写此方法进行自己的初始化逻辑，
     * 如加载配置文件、初始化内部状态、注册监听器等。
     * <p>
     * 默认实现为空，子类根据需要选择性重写。
     * <p>
     * 注意：此时getContext()已可正常使用。
     */
    protected void init() {
        // 默认空实现，子类可根据需要重写
    }

    /**
     * 获取插件上下文实例
     * <p>
     * 返回在init()方法中设置的PluginContext实例，保证不为null。
     * 如果在init()调用前访问此方法，可能会抛出空指针异常。
     *
     * @return 插件上下文实例，不会为null（init调用后）
     */
    @NonNull
    @Override
    public PluginContext getContext() {
        return Objects.requireNonNull(context);
    }

    /**
     * 判断当前菜单是否启用的默认实现
     * <p>
     * 基类提供的默认实现始终返回 {@code true}，表示菜单默认处于启用状态。
     * 这适用于大多数常规菜单功能的场景。
     * <p>
     * 子类可以重写此方法以实现自定义的启用逻辑，重写示例：
     * <pre>{@code
     * @Override
     * public boolean isEnabled() {
     *     // 示例：根据用户设置决定是否启用
     *     return getContext().getPreferences().getBoolean("enable_my_menu", true);
     * }
     * }</pre>
     *
     * @return 默认返回 {@code true}，表示菜单启用
     */
    @Override
    public boolean isEnabled() {
        return true;
    }

    /**
     * 插件按钮点击的默认实现：使用对话框显示插件名、插件ID、接口名
     * <p>
     * 在菜单编辑界面，每个插件功能项的右边有一个插件图标样式的按钮，当该按钮被点击后会调用此方法。
     *
     * @param pluginUI 插件UI接口
     */
    @Override
    public void onPluginButtonClick(@NonNull PluginUI pluginUI) {
        PluginContext context = getContext();
        String pluginId = context.getPluginId();
        int idLen = pluginId.length();
        SpannableString message = new SpannableString(pluginId + "\n\n" + getClass().getName());
        message.setSpan(new RelativeSizeSpan(0.1f), idLen + 1, idLen + 2, Spanned.SPAN_INCLUSIVE_EXCLUSIVE); // 增加行距
        message.setSpan(new ForegroundColorSpan(pluginUI.colorTextSecondary()), idLen + 2, message.length(), Spanned.SPAN_INCLUSIVE_EXCLUSIVE);
        message.setSpan(new RelativeSizeSpan(0.8f), idLen + 2, message.length(), Spanned.SPAN_INCLUSIVE_EXCLUSIVE);
        pluginUI.buildDialog()
                .setTitle(context.getPluginName())
                .setMessage(message)
                .setPositiveButton("{close}", null)
                .show();
    }
}
