package bin.mt.plugin.api;

import java.util.Set;

/**
 * 多国语言支持接口
 * 用于实现插件的国际化功能，支持多种语言文本的获取和管理
 * 使用方法请参考官方示例文档
 */
public interface LocalString {

    /**
     * 获取当前语言包的名称
     * <p>
     * 例如文件<code>String.mtl</code>与<code>String-zh-CN.mtl</code>组成一个语言包，
     * 它们语言包的名称就是<code>String</code>
     *
     * @return 语言包名称
     */
    String getName();

    /**
     * 根据键值获取当前本地语言对应的文本内容
     * 使用系统当前设置的语言环境
     *
     * @param key 文本键值，不能为null
     * @return 对应的本地化文本，如果key不存在则返回key本身
     */
    String get(String key);

    /**
     * 根据键值获取当前本地语言对应的文本内容，并支持格式化参数
     * 使用系统当前设置的语言环境，支持String.format()风格的参数替换
     *
     * @param key        文本键值，不能为null
     * @param formatArgs 格式化参数，用于替换文本中的占位符（如%s、%d等）
     * @return 格式化后的本地化文本，如果key不存在则返回key本身
     */
    String get(String key, Object... formatArgs);

    /**
     * 根据键值获取指定语言的文本内容
     *
     * @param key          文本键值，不能为null
     * @param languageCode 指定的语言代码，例如"zh-CN"、"en"等
     * @return 对应的文本内容，如果key不存在则返回key本身
     */
    String get(String key, String languageCode);

    /**
     * 根据键值获取本地语言文本，当键值不存在时返回默认值
     * 注意：一般不建议使用此方法，推荐使用get(String key)方法
     *
     * @param key 文本键值
     * @param def 默认值，当key不存在时返回此值
     * @return 对应的本地化文本或默认值
     */
    String getOrDefault(String key, String def);

    /**
     * 根据键值获取指定语言的文本内容，当键值不存在时返回默认值
     * 注意：一般不建议使用此方法，推荐使用get(String key, String languageCode)方法
     *
     * @param key          文本键值
     * @param def          默认值，当key不存在时返回此值
     * @param languageCode 指定的语言代码
     * @return 对应的文本内容或默认值
     */
    String getOrDefault(String key, String def, String languageCode);

    /**
     * 获取默认语言包中所有可用的键值集合
     *
     * @return 包含所有键值的Set集合，不会返回null
     */
    Set<String> keySet();

}
