package bin.mt.plugin.api.ui.dialog;

import android.text.TextUtils;

import bin.mt.plugin.api.ui.PluginProgressBar;
import bin.mt.plugin.api.ui.PluginTextView;
import bin.mt.plugin.api.ui.PluginUI;
import bin.mt.plugin.api.ui.PluginView;
import bin.mt.plugin.api.util.ThreadUtil;
import bin.mt.plugin.api.util.UIUpdater;
import bin.mt.plugin.api.util.UIUpdaterGroup;

/**
 * 进度对话框类，用于显示带有进度条的对话框
 */
public class ProgressDialog extends BaseCancelableDialog<ProgressDialog> {
    private final PluginDialog dialog;
    private final PluginTextView messageView;
    private final PluginProgressBar progressBar;
    private final UIUpdater<CharSequence> titleUpdater;
    private final UIUpdater<CharSequence> messageUpdater;
    private final UIUpdater<Integer> progressUpdater;
    private final UIUpdater<Integer> secondaryProgressUpdater;

    /**
     * 构造函数，创建进度对话框
     *
     * @param pluginUI 插件UI实例，用于构建对话框界面
     */
    public ProgressDialog(PluginUI pluginUI) {
        super(pluginUI);
        // 构建垂直布局的对话框视图
        PluginView pluginView = pluginUI.buildVerticalLayout()
                .paddingHorizontal(pluginUI.dialogPaddingHorizontal())
                .paddingBottom(pluginUI.dialogPaddingVertical())
                .addTextView("message").textSize(16).singleLine().ellipsize(TextUtils.TruncateAt.END).marginTopDp(8).gone()
                .addProgressBar("progressBar").marginTopDp(12)
                .build();

        // 创建不可取消的对话框
        dialog = pluginUI.buildDialog().setView(pluginView).setCancelable(false).create();

        // 获取视图组件引用
        messageView = pluginView.requireViewById("message");
        progressBar = pluginView.requireViewById("progressBar");

        // 创建UI更新器组，用于线程安全的UI更新
        UIUpdaterGroup updaterGroup = new UIUpdaterGroup();
        titleUpdater = updaterGroup.registerUpdater(dialog::setTitle);
        messageUpdater = updaterGroup.registerUpdater(text -> {
            if (text == null) {
                if (messageView.getVisibility() == PluginView.VISIBLE) {
                    messageView.setGone();
                    progressBar.setMarginTopDp(12);
                }
            } else {
                if (messageView.getVisibility() == PluginView.GONE) {
                    messageView.setVisible();
                    progressBar.setMarginTop(6);
                }
            }
            messageView.setText(text);
        });
        progressUpdater = updaterGroup.registerUpdater(progressBar::setProgress);
        secondaryProgressUpdater = updaterGroup.registerUpdater(progressBar::setSecondaryProgress);
    }

    /**
     * 设置标题文本
     *
     * @param title 要显示的标题文本
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setTitle(CharSequence title) {
        titleUpdater.submitUpdate(title);
        return this;
    }

    /**
     * 设置消息文本
     *
     * @param message 要显示的消息文本，传入null时会隐藏消息视图
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setMessage(CharSequence message) {
        messageUpdater.submitUpdate(message);
        return this;
    }

    /**
     * 设置进度值
     *
     * @param progress 进度值
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setProgress(int progress) {
        progressUpdater.submitUpdate(progress);
        return this;
    }

    /**
     * 设置次要进度值
     *
     * @param secondaryProgress 次要进度值
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setSecondaryProgress(int secondaryProgress) {
        secondaryProgressUpdater.submitUpdate(secondaryProgress);
        return this;
    }

    /**
     * 设置进度条的最大值
     *
     * @param maxProgress 进度条最大值
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setMaxProgress(int maxProgress) {
        ThreadUtil.runOnUiThread(() -> progressBar.setMaxProgress(maxProgress));
        return this;
    }

    /**
     * 设置进度条为不确定进度模式
     *
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setIndeterminate() {
        setIndeterminate(true);
        return this;
    }

    /**
     * 设置进度条是否为不确定进度模式
     *
     * @param indeterminate true设置为不确定进度模式，false设置为确定进度模式
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog setIndeterminate(boolean indeterminate) {
        if (progressBar.isIndeterminate() != indeterminate) {
            ThreadUtil.runOnUiThread(() -> progressBar.setIndeterminate(indeterminate));
        }
        return this;
    }

    /**
     * 获取消息视图
     *
     * @return 消息文本视图实例
     */
    public PluginTextView getMessageView() {
        return messageView;
    }

    /**
     * 获取进度条视图
     *
     * @return 进度条实例
     */
    public PluginProgressBar getProgressBar() {
        return progressBar;
    }

    @Override
    protected PluginDialog getDialog() {
        return dialog;
    }

    /**
     * 显示对话框
     * 注意：此方法必须在UI线程中调用
     *
     * @return 返回当前ProgressDialog实例
     */
    public ProgressDialog show() {
        ThreadUtil.assertInUIThread(); // 确保在UI线程中执行
        dialog.show();
        return this;
    }

    /**
     * 检查对话框是否正在显示
     *
     * @return 如果对话框正在显示返回true，否则返回false
     */
    public boolean isShowing() {
        return dialog.isShowing();
    }

    /**
     * 关闭对话框
     * 此方法会自动切换到UI线程执行，确保线程安全
     */
    @Override
    public void dismiss() {
        ThreadUtil.runOnUiThread(() -> {
            if (dialog.isShowing()) {
                dialog.dismiss();
            }
        });
    }
}
