package bin.mt.plugin.api.ui.builder;

import android.content.res.ColorStateList;
import android.graphics.Typeface;
import android.text.TextUtils;

import bin.mt.plugin.api.ui.PluginView;

/**
 * 插件文本视图构建器基础接口
 * <p>
 * 该接口为所有文本相关的UI组件提供基础的文本属性设置功能，
 * 包括文本内容、字体大小、颜色、样式、对齐方式等。
 *
 * @param <T> 具体的构建器类型，使用泛型实现链式调用的类型安全
 * @see PluginBaseViewBuilder
 * @see PluginTextViewBuilder
 * @see PluginBaseButtonBuilder
 */
public interface PluginBaseTextViewBuilder<T extends PluginViewBuilder> extends PluginBaseViewBuilder<T> {

    /**
     * 设置文本内容
     * <p>
     * 如果text类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本
     * <p>
     * 支持的key格式：
     * <ul>
     * <li><b>"{key}"</b>：优先从strings语言包中获取key对应词条，若未找到则回退到MT内置strings语言包中获取</li>
     * <li><b>"{pack:key}"</b>：直接从指定的pack语言包中获取key对应词条，没有回退机制</li>
     * </ul>
     * <p>
     * 使用示例：
     * <pre>{@code
     *   text("{plugin_name}")  // 从strings包获取
     *   text("{common:ok}")    // 从common包获取
     * }</pre>
     * <p>
     * 在一些特殊情况下若不想被转化为本地化文本，可以传入非String类型，例如：
     * <code>text(new SpannedString("{key}"))</code>
     *
     * @param text 文本内容
     * @return 当前构建器实例，支持链式调用
     */
    T text(CharSequence text);

    /**
     * 设置文本字体大小
     *
     * @param sizeSp 字体大小，单位为sp
     * @return 当前构建器实例，支持链式调用
     */
    T textSize(float sizeSp);

    /**
     * 设置文本颜色，在任何状态下均为该颜色
     *
     * @param color 颜色值，格式为ARGB
     * @return 当前构建器实例，支持链式调用
     */
    T textColor(int color);

    /**
     * 设置启用和禁用状态下的文本颜色
     *
     * @param enabledColor 启用状态下的文本颜色，格式为ARGB
     * @param disableColor 禁用状态下的文本颜色，格式为ARGB
     * @return 当前构建器实例，支持链式调用
     *
     * @see #textColor(ColorStateList)
     * @see PluginView#isEnabled()
     * @see PluginView#setEnabled(boolean)
     * @see android.R.attr#state_enabled
     */
    T textColor(int enabledColor, int disabledColor);

    /**
     * 使用ColorStateList设置文本颜色。
     * ColorStateList允许根据控件的不同状态（如pressed、focused、selected、enabled等）
     * 显示不同的颜色，提供更丰富的视觉反馈
     *
     * @param colors ColorStateList对象，定义了不同状态下的文本颜色
     * @return 当前构建器实例，支持链式调用
     *
     * @see ColorStateList
     * @see #textColor(int, int)
     * @see android.R.attr#state_pressed
     * @see android.R.attr#state_focused
     * @see android.R.attr#state_selected
     * @see android.R.attr#state_enabled
     */
    T textColor(ColorStateList textColor);

    /**
     * 设置文本对齐方式
     *
     * @param gravity 对齐方式，使用Gravity常量，如Gravity.CENTER、Gravity.LEFT等
     * @return 当前构建器实例，支持链式调用
     */
    T textGravity(int gravity);

    /**
     * 设置文本样式
     * <p>
     * 此方法会覆盖typeface()的调用结果
     *
     * @param bold   是否加粗显示
     * @param italic 是否斜体显示
     * @return 当前构建器实例，支持链式调用
     */
    T textStyle(boolean bold, boolean italic);

    /**
     * 设置文本样式为加粗
     * <p>
     * 相当于调用<code>textStyle(true, false)</code>，
     * 如果先调用 bold() 再调用 italic()，最终效果只有斜体没有加粗
     *
     * @return 当前构建器实例，支持链式调用
     */
    T bold();

    /**
     * 设置文本样式为斜体
     * <p>
     * 相当于调用<code>textStyle(false, true)</code>，
     * 如果先调用 italic() 再调用 bold()，最终效果只有加粗没有斜体
     *
     * @return 当前构建器实例，支持链式调用
     */
    T italic();

    /**
     * 设置字体类型
     * <p>
     * 此方法会覆盖textStyle()的调用结果
     *
     * @param typeface 字体类型，如Typeface.DEFAULT、Typeface.SERIF等
     * @return 当前构建器实例，支持链式调用
     */
    T typeface(Typeface typeface);

    /**
     * 设置行间距
     *
     * @param add  额外增加的行间距，单位为px
     * @param mult 行间距倍数，1.0表示默认行间距
     * @return 当前构建器实例，支持链式调用
     */
    T lineSpacing(float add, float mult);

    /**
     * 设置单行模式
     * <p>
     * 与调用 lines(1) 的区别是换行符会被当作空格显示，例如对于字符串 "line\nline\nline"
     * <ul>
     * <li>singleLine() 显示为 "line line line"</li>
     * <li>lines(1) 显示为 "line"</li>
     * </ul>
     *
     * @return 当前构建器实例，支持链式调用
     */
    T singleLine();

    /**
     * 设置显示行数
     *
     * @param lines 固定显示的行数
     * @return 当前构建器实例，支持链式调用
     */
    T lines(int lines);

    /**
     * 设置最大显示行数
     *
     * @param maxLines 最多显示的行数，超出部分会被截断
     * @return 当前构建器实例，支持链式调用
     */
    T maxLines(int maxLines);

    /**
     * 设置最小显示行数
     *
     * @param minLines 最少显示的行数，不足时会用空行补充
     * @return 当前构建器实例，支持链式调用
     */
    T minLines(int minLines);

    /**
     * 设置文本截断方式
     * <p>
     * 当文本超出显示区域时的截断处理方式
     *
     * @param truncate 截断方式，如TextUtils.TruncateAt.END表示末尾显示省略号
     * @return 当前构建器实例，支持链式调用
     */
    T ellipsize(TextUtils.TruncateAt truncate);

    /**
     * 设置文本在显示时是否强制大写
     *
     * @param allCaps 是否强制大写
     * @return 当前构建器实例，支持链式调用
     */
    T allCaps(boolean allCaps);
}
