package bin.mt.plugin.api.ui;

import android.content.res.ColorStateList;
import android.graphics.drawable.Drawable;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import bin.mt.plugin.api.PluginContext;
import bin.mt.plugin.api.preference.PluginPreference;
import bin.mt.plugin.api.ui.builder.PluginBaseButtonBuilder;
import bin.mt.plugin.api.ui.builder.PluginBaseCompoundButtonBuilder;
import bin.mt.plugin.api.ui.builder.PluginBaseLinearLayoutBuilder;
import bin.mt.plugin.api.ui.builder.PluginBaseTextViewBuilder;
import bin.mt.plugin.api.ui.builder.PluginBaseViewBuilder;
import bin.mt.plugin.api.ui.builder.PluginBaseViewGroupBuilder;
import bin.mt.plugin.api.ui.builder.PluginButtonBuilder;
import bin.mt.plugin.api.ui.builder.PluginCheckBoxBuilder;
import bin.mt.plugin.api.ui.builder.PluginEditTextBuilder;
import bin.mt.plugin.api.ui.builder.PluginLinearLayoutBuilder;
import bin.mt.plugin.api.ui.builder.PluginProgressBarBuilder;
import bin.mt.plugin.api.ui.builder.PluginRadioButtonBuilder;
import bin.mt.plugin.api.ui.builder.PluginRadioGroupBuilder;
import bin.mt.plugin.api.ui.builder.PluginRootLayoutBuilder;
import bin.mt.plugin.api.ui.builder.PluginSpinnerBuilder;
import bin.mt.plugin.api.ui.builder.PluginSwitchButtonBuilder;
import bin.mt.plugin.api.ui.builder.PluginTextViewBuilder;
import bin.mt.plugin.api.ui.dialog.PluginDialog;

/**
 * 插件UI功能的核心接口
 * <p>
 * 可用于创建UI组件、显示对话框、跳转设置界面等
 * <p>
 * <strong>使用示例：</strong>
 * <pre>{@code
 * PluginView view = pluginUI.buildVerticalLayout()
 *   .addTextView().text("Hello World")
 *   .addButton().text("Click Me")
 *   .build();
 * }</pre>
 */
public interface PluginUI {

    /**
     * 获取插件上下文对象
     *
     * @return PluginContext 插件运行时上下文
     */
    PluginContext getContext();

    /**
     * 相当于调用 <code>getContext().showToast(msg)</code>
     *
     * @see PluginContext#showToast(CharSequence)
     */
    void showToast(CharSequence msg);

    /**
     * 相当于调用 <code>getContext().showToast(msg, formatArgs)</code>
     *
     * @see PluginContext#showToast(CharSequence, Object...)
     */
    void showToast(CharSequence msg, Object... formatArgs);

    /**
     * 相当于调用 <code>getContext().showToastL(msg)</code>
     *
     * @see PluginContext#showToastL(CharSequence)
     */
    void showToastL(CharSequence msg);

    /**
     * 相当于调用 <code>getContext().showToastL(msg, formatArgs)</code>
     *
     * @see PluginContext#showToastL(CharSequence, Object...)
     */
    void showToastL(CharSequence msg, Object... formatArgs);

    /**
     * 取消当前正在显示的Toast提示消息
     */
    void cancelToast();

    /**
     * 禁用ID严格模式
     *
     * @return 当前PluginUI实例，支持链式调用
     * @see #isStrictIdModeEnabled()
     */
    PluginUI disableStrictIdMode();

    /**
     * 设置是否启用ID严格模式
     *
     * @param enabled 是否启用ID严格模式
     * @return 当前PluginUI实例，支持链式调用
     * @see #isStrictIdModeEnabled()
     */
    PluginUI setStrictIdModeEnabled(boolean enabled);

    /**
     * 获取当前是否启用ID严格模式（默认启用）
     * <p>
     * 在ID严格模式下，同一个根布局构建器实例下的所有PluginView的id不能重复，
     * 如果出现重复将在配置id时抛出异常
     *
     * @return 是否启用ID严格模式
     */
    boolean isStrictIdModeEnabled();

    /**
     * 设置默认样式
     * 该样式将应用到所有通过此PluginUI创建的组件
     *
     * @param style 要应用的样式，不能为null
     * @return 当前PluginUI实例，支持链式调用
     */
    PluginUI defaultStyle(@NonNull Style style);

    /**
     * 创建以水平线形布局为底的根布局构建器
     *
     * @return PluginRootLayoutBuilder 根布局构建器实例
     */
    PluginRootLayoutBuilder buildHorizontalLayout();

    /**
     * 创建以垂直线形布局为底的根布局构建器
     *
     * @return PluginRootLayoutBuilder 根布局构建器实例
     */
    PluginRootLayoutBuilder buildVerticalLayout();

    /**
     * 创建以帧布局为底的根布局构建器
     *
     * @return PluginRootLayoutBuilder 根布局构建器实例
     */
    PluginRootLayoutBuilder buildFrameLayout();

    /**
     * 创建对话框构建器
     *
     * @return PluginDialog.Builder 对话框构建器实例
     */
    PluginDialog.Builder buildDialog();

    /**
     * 弹出一个自定义标题和内容的对话框，并包含一个关闭按钮
     * <p>
     * 如果title或message类型为String且为<code>{key}</code>格式，将尝试转化为本地化文本，
     * 具体请参考 {@link PluginBaseTextViewBuilder#text(CharSequence)}
     * <p>
     * 标题可选用内置语言包提供的词条：{tip}、{notice}、{information}、{warning}、{error}
     *
     * @param title   对话框标题内容
     * @param message 对话框消息内容
     * @see PluginBaseTextViewBuilder#text(CharSequence)
     */
    void showMessage(CharSequence title, CharSequence message);

    /**
     * 弹出一个对话框显示错误信息，同时支持查看错误详情功能，传入的异常还会被记录到插件日志中
     *
     * @param e 捕获到的异常
     */
    void showErrorMessage(Throwable e);

    /**
     * 跳转并显示设置界面
     *
     * @param clazz 设置界面类，需实现PluginPreference接口，为null则表示主设置界面类
     * @see PluginPreference
     */
    void showPreference(@Nullable Class<? extends PluginPreference> clazz);

    /**
     * @return 当前是否为深色主题（夜间模式）
     */
    boolean isDarkTheme();

    //
    // 下面是获取颜色的相关方法，具体颜色值会根据MT当前主题、是否夜间模式而有所不同
    //

    /**
     * @return 主要颜色（Primary Color）
     */
    int colorPrimary();

    /**
     * @return 获取强调色（Accent Color）
     */
    int colorAccent();

    /**
     * @return 分割线颜色
     */
    int colorDivider();

    /**
     * @return 错误颜色
     */
    int colorError();

    /**
     * @return 警告颜色
     */
    int colorWarning();

    /**
     * @return 主要文本颜色
     */
    int colorText();

    /**
     * @return 次要文本颜色。相比主要文本颜色，次要文本颜色较浅
     */
    int colorTextSecondary();

    /**
     * @return 带状态的主要文本颜色
     */
    ColorStateList colorTextStateList();

    /**
     * @return 带状态的次要文本颜色。相比主要文本颜色，次要文本颜色较浅
     */
    ColorStateList colorTextSecondaryStateList();

    /**
     * @return 对话框左右内边距推荐值，单位为px
     */
    int dialogPaddingHorizontal();

    /**
     * @return 对话框上下内边距推荐值，单位为px，如果设置了对话框按钮，则不推荐设置底部内边距
     */
    int dialogPaddingVertical();

    /**
     * 用于可选择项目背景，例如PluginTextView设置点击事件后，按下没有视觉效果，可调用此方法设置其背景
     *
     * @return 可选择项目背景drawable
     */
    Drawable selectableItemBackground();

    /**
     * @return 无边框可选择项目背景drawable
     */
    Drawable selectableItemBackgroundBorderless();

    /**
     * 将dp值转换为像素值
     *
     * @param dp 要转换的dp值
     * @return 对应的像素值
     */
    int dp2px(float dp);

    /**
     * 将sp值转换为像素值
     *
     * @param sp 要转换的dp值
     * @return 对应的像素值
     */
    int sp2px(float sp);

    /**
     * 获取当前使用的样式
     *
     * @return 非空的Style实例
     */
    @NonNull
    Style getStyle();

    /**
     * 默认样式实现
     */
    Style DEFAULT_STYLE = new Style() {

        @Override
        protected void handleRootLayout(PluginUI pluginUI, PluginRootLayoutBuilder builder) {
            // 插件创建的View基本都用在对话框中显示
            builder.paddingHorizontal(pluginUI.dialogPaddingHorizontal());
        }

        @Override
        protected void handleTextView(PluginUI pluginUI, PluginTextViewBuilder builder) {
            // 设置默认的文字颜色与大小
            builder.textColor(pluginUI.colorTextStateList());
            builder.textSize(16);
        }

        @Override
        protected void handleEditText(PluginUI pluginUI, PluginEditTextBuilder builder) {
            // 默认的宽度和高度都是WRAP_CONTENT，对于编辑框一般使用MATCH_PARENT
            builder.widthMatchParent();
            // box风格的编辑框一般用于多行文本输入
            if (builder.isBoxStyle()) {
                builder.inputTypeMultiline();
            }
        }

        @Override
        protected void handleLinearLayout(PluginUI pluginUI, PluginLinearLayoutBuilder builder) {
            // 默认的宽度和高度都是WRAP_CONTENT，对于水平布局一般使用MATCH_PARENT
            if (builder.isHorizontal()) {
                builder.widthMatchParent();
            }
        }

        @Override
        protected void handleRadioGroup(PluginUI pluginUI, PluginRadioGroupBuilder builder) {
            // 默认的宽度和高度都是WRAP_CONTENT，对于水平布局一般使用MATCH_PARENT
            if (builder.isHorizontal()) {
                builder.widthMatchParent();
            }
        }

    };

    /**
     * 样式配置类
     * <p>
     * 通过继承此类并重写相应方法，可以自定义各种UI组件的默认样式
     * 该类使用分发模式，根据组件类型调用相应的处理方法
     * </p>
     * <p>
     * dispatch方法会在builder创建之后立刻调用，例如addTextView()的内部实现代码大致如下：
     * <pre>{@code
     * PluginTextViewBuilder addTextView() {
     *     PluginTextViewBuilder builder = generateTextViewBuilder();
     *     getStyle().dispatch(pluginUI, builder);
     *     return builder;
     * }
     * }</pre>
     * </p>
     * 建议只对 builder 进行样式修改相关的操作，MT不会阻止你调用builder.addTextView()之类的方法，除非你想进行一些骚操作 :)
     */
    class Style {

        /**
         * 样式分发入口方法
         * 根据builder类型分发到相应的处理方法
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  要应用样式的构建器
         */
        public final void dispatch(PluginUI pluginUI, PluginBaseViewBuilder<?> builder) {
            if (handleBaseView(pluginUI, builder)) {
                return;
            }
            if (builder instanceof PluginBaseTextViewBuilder) {
                dispatchBaseTextView(pluginUI, (PluginBaseTextViewBuilder<?>) builder);
            } else if (builder instanceof PluginBaseViewGroupBuilder) {
                dispatchBaseViewGroup(pluginUI, (PluginBaseViewGroupBuilder<?>) builder);
            } else if (builder instanceof PluginEditTextBuilder) {
                handleEditText(pluginUI, (PluginEditTextBuilder) builder);
            } else if (builder instanceof PluginSpinnerBuilder) {
                handleSpinner(pluginUI, (PluginSpinnerBuilder) builder);
            } else if (builder instanceof PluginProgressBarBuilder) {
                handleProgressBar(pluginUI, (PluginProgressBarBuilder) builder);
            } else if (builder instanceof PluginRootLayoutBuilder) {
                handleRootLayout(pluginUI, (PluginRootLayoutBuilder) builder);
            } else {
                throw new RuntimeException(builder.getClass().getSimpleName());
            }
        }

        /**
         * 分发BaseTextView相关的样式处理
         */
        protected final void dispatchBaseTextView(PluginUI pluginUI, PluginBaseTextViewBuilder<?> builder) {
            if (handleBaseTextView(pluginUI, builder)) {
                return;
            }
            if (builder instanceof PluginBaseButtonBuilder) {
                dispatchBaseButton(pluginUI, (PluginBaseButtonBuilder<?>) builder);
            } else if (builder instanceof PluginTextViewBuilder) {
                handleTextView(pluginUI, (PluginTextViewBuilder) builder);
            } else {
                throw new RuntimeException(builder.getClass().getSimpleName());
            }
        }

        /**
         * 分发BaseButton相关的样式处理
         */
        protected final void dispatchBaseButton(PluginUI pluginUI, PluginBaseButtonBuilder<?> builder) {
            if (handleBaseButton(pluginUI, builder)) {
                return;
            }
            if (builder instanceof PluginBaseCompoundButtonBuilder) {
                dispatchBaseCompoundButton(pluginUI, (PluginBaseCompoundButtonBuilder<?>) builder);
            } else if (builder instanceof PluginButtonBuilder) {
                handleButton(pluginUI, (PluginButtonBuilder) builder);
            } else {
                throw new RuntimeException(builder.getClass().getSimpleName());
            }
        }

        /**
         * 分发BaseCompoundButton相关的样式处理
         */
        protected final void dispatchBaseCompoundButton(PluginUI pluginUI, PluginBaseCompoundButtonBuilder<?> builder) {
            if (handleBaseCompoundButton(pluginUI, builder)) {
                return;
            }
            if (builder instanceof PluginCheckBoxBuilder) {
                handleCheckBox(pluginUI, (PluginCheckBoxBuilder) builder);
            } else if (builder instanceof PluginSwitchButtonBuilder) {
                handleSwitchButton(pluginUI, (PluginSwitchButtonBuilder) builder);
            } else if (builder instanceof PluginRadioButtonBuilder) {
                handleRadioButton(pluginUI, (PluginRadioButtonBuilder) builder);
            } else {
                throw new RuntimeException(builder.getClass().getSimpleName());
            }
        }

        /**
         * 分发BaseViewGroup相关的样式处理
         */
        protected final void dispatchBaseViewGroup(PluginUI pluginUI, PluginBaseViewGroupBuilder<?> builder) {
            if (handleBaseViewGroup(pluginUI, builder)) {
                return;
            }
            if (builder instanceof PluginBaseLinearLayoutBuilder) {
                dispatchBaseLinearLayout(pluginUI, (PluginBaseLinearLayoutBuilder<?>) builder);
            } else {
                throw new RuntimeException(builder.getClass().getSimpleName());
            }
        }

        protected final void dispatchBaseLinearLayout(PluginUI pluginUI, PluginBaseLinearLayoutBuilder<?> builder) {
            if (builder instanceof PluginLinearLayoutBuilder) {
                handleLinearLayout(pluginUI, (PluginLinearLayoutBuilder) builder);
            } else if (builder instanceof PluginRadioGroupBuilder) {
                handleRadioGroup(pluginUI, (PluginRadioGroupBuilder) builder);
            } else {
                throw new RuntimeException(builder.getClass().getSimpleName());
            }
        }

        /**
         * 处理所有视图的基础样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  视图构建器
         * @return true表示已完全处理，不需要继续分发；false表示需要继续分发到子类型
         */
        protected boolean handleBaseView(PluginUI pluginUI, PluginBaseViewBuilder<?> builder) {
            return false;
        }

        /**
         * 处理文本视图的基础样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  文本视图构建器
         * @return true表示已完全处理，不需要继续分发；false表示需要继续分发到子类型
         */
        protected boolean handleBaseTextView(PluginUI pluginUI, PluginBaseTextViewBuilder<?> builder) {
            return false;
        }

        /**
         * 处理按钮的基础样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  按钮构建器
         * @return true表示已完全处理，不需要继续分发；false表示需要继续分发到子类型
         */
        protected boolean handleBaseButton(PluginUI pluginUI, PluginBaseButtonBuilder<?> builder) {
            return false;
        }

        /**
         * 处理复合按钮的基础样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  复合按钮构建器
         * @return true表示已完全处理，不需要继续分发；false表示需要继续分发到子类型
         */
        protected boolean handleBaseCompoundButton(PluginUI pluginUI, PluginBaseCompoundButtonBuilder<?> builder) {
            return false;
        }

        /**
         * 处理视图组的基础样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  视图组构建器
         * @return true表示已完全处理，不需要继续分发；false表示需要继续分发到子类型
         */
        protected boolean handleBaseViewGroup(PluginUI pluginUI, PluginBaseViewGroupBuilder<?> builder) {
            return false;
        }

        /**
         * 处理文本视图样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  文本视图构建器
         */
        protected void handleTextView(PluginUI pluginUI, PluginTextViewBuilder builder) {

        }

        /**
         * 处理按钮样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  按钮构建器
         */
        protected void handleButton(PluginUI pluginUI, PluginButtonBuilder builder) {

        }

        /**
         * 处理复选框样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  复选框构建器
         */
        protected void handleCheckBox(PluginUI pluginUI, PluginCheckBoxBuilder builder) {

        }

        /**
         * 处理开关按钮样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  开关按钮构建器
         */
        protected void handleSwitchButton(PluginUI pluginUI, PluginSwitchButtonBuilder builder) {

        }

        /**
         * 处理单选按钮样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  单选按钮构建器
         */
        protected void handleRadioButton(PluginUI pluginUI, PluginRadioButtonBuilder builder) {

        }

        /**
         * 处理下拉选择器样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  下拉选择器构建器
         */
        protected void handleSpinner(PluginUI pluginUI, PluginSpinnerBuilder builder) {

        }

        /**
         * 处理进度条样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  进度条构建器
         */
        protected void handleProgressBar(PluginUI pluginUI, PluginProgressBarBuilder builder) {

        }

        /**
         * 处理输入框样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  输入框构建器
         */
        protected void handleEditText(PluginUI pluginUI, PluginEditTextBuilder builder) {

        }

        /**
         * 处理线性布局样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  线性布局构建器
         */
        protected void handleLinearLayout(PluginUI pluginUI, PluginLinearLayoutBuilder builder) {

        }

        /**
         * 处理单选按钮组
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  单选按钮组构建器
         */
        protected void handleRadioGroup(PluginUI pluginUI, PluginRadioGroupBuilder builder) {

        }

        /**
         * 处理根布局样式
         *
         * @param pluginUI 当前的PluginUI实例
         * @param builder  根布局构建器
         */
        protected void handleRootLayout(PluginUI pluginUI, PluginRootLayoutBuilder builder) {

        }

        /**
         * 样式修改器类
         * <p>
         * 提供一个便捷的内部类，用于创建基于当前样式的修改器。
         * 使用装饰器模式包装当前样式实例，允许在不修改原有样式的基础上
         * 进行额外的样式定制。
         * </p>
         * <p>
         * 使用示例：
         * <pre>{@code
         * PluginUI.Style newStyle = pluginUi.getStyle().new Modifier() {
         *     // override methods
         * };
         * }</pre>
         * </p>
         */
        public abstract class Modifier extends Style {

            /**
             * 被包装的样式实例
             */
            private final Style style = Style.this;

            public Modifier() {
            }

            @Override
            protected boolean handleBaseView(PluginUI pluginUI, PluginBaseViewBuilder<?> builder) {
                return style.handleBaseView(pluginUI, builder);
            }

            @Override
            protected boolean handleBaseTextView(PluginUI pluginUI, PluginBaseTextViewBuilder<?> builder) {
                return style.handleBaseTextView(pluginUI, builder);
            }

            @Override
            protected boolean handleBaseButton(PluginUI pluginUI, PluginBaseButtonBuilder<?> builder) {
                return style.handleBaseButton(pluginUI, builder);
            }

            @Override
            protected boolean handleBaseCompoundButton(PluginUI pluginUI, PluginBaseCompoundButtonBuilder<?> builder) {
                return style.handleBaseCompoundButton(pluginUI, builder);
            }

            @Override
            protected boolean handleBaseViewGroup(PluginUI pluginUI, PluginBaseViewGroupBuilder<?> builder) {
                return style.handleBaseViewGroup(pluginUI, builder);
            }

            @Override
            protected void handleTextView(PluginUI pluginUI, PluginTextViewBuilder builder) {
                style.handleTextView(pluginUI, builder);
            }

            @Override
            protected void handleButton(PluginUI pluginUI, PluginButtonBuilder builder) {
                style.handleButton(pluginUI, builder);
            }

            @Override
            protected void handleCheckBox(PluginUI pluginUI, PluginCheckBoxBuilder builder) {
                style.handleCheckBox(pluginUI, builder);
            }

            @Override
            protected void handleSwitchButton(PluginUI pluginUI, PluginSwitchButtonBuilder builder) {
                style.handleSwitchButton(pluginUI, builder);
            }

            @Override
            protected void handleRadioButton(PluginUI pluginUI, PluginRadioButtonBuilder builder) {
                style.handleRadioButton(pluginUI, builder);
            }

            @Override
            protected void handleRadioGroup(PluginUI pluginUI, PluginRadioGroupBuilder builder) {
                style.handleRadioGroup(pluginUI, builder);
            }

            @Override
            protected void handleSpinner(PluginUI pluginUI, PluginSpinnerBuilder builder) {
                style.handleSpinner(pluginUI, builder);
            }

            @Override
            protected void handleProgressBar(PluginUI pluginUI, PluginProgressBarBuilder builder) {
                style.handleProgressBar(pluginUI, builder);
            }

            @Override
            protected void handleEditText(PluginUI pluginUI, PluginEditTextBuilder builder) {
                style.handleEditText(pluginUI, builder);
            }

            @Override
            protected void handleLinearLayout(PluginUI pluginUI, PluginLinearLayoutBuilder builder) {
                style.handleLinearLayout(pluginUI, builder);
            }

            @Override
            protected void handleRootLayout(PluginUI pluginUI, PluginRootLayoutBuilder builder) {
                style.handleRootLayout(pluginUI, builder);
            }
        }

    }

}
