package bin.mt.plugin.api.translation;

import androidx.annotation.NonNull;

import java.io.IOException;
import java.util.List;

import bin.mt.plugin.api.PluginContext;

/**
 * 翻译引擎接口
 * 定义了翻译引擎的核心功能和生命周期方法
 */
public interface TranslationEngine {

    /**
     * 初始化翻译引擎，每个实例仅调用一次
     *
     * @param context MT插件上下文
     */
    void init(PluginContext context);

    /**
     * 获取初始化时传入的上下文
     *
     * @return init中传入的context
     */
    PluginContext getContext();

    /**
     * 获取翻译引擎名称
     * <p>
     * 如果返回的名称为{key}格式，将会自动调用
     * getContext().getLocalString().getOrDefault("key", "{key}") 尝试转化为本地化文本
     *
     * @return 翻译引擎名称
     */
    @NonNull
    String name();

    /**
     * 获取翻译引擎配置信息
     *
     * @return 配置信息
     */
    @NonNull
    Configuration getConfiguration();

    /**
     * 加载支持的源语言列表
     *
     * @return 源语言列表
     */
    @NonNull
    List<String> loadSourceLanguages();

    /**
     * 根据源语言加载对应的目标语言列表
     *
     * @param sourceLanguage 源语言名称
     * @return 目标语言列表
     */
    @NonNull
    List<String> loadTargetLanguages(String sourceLanguage);

    /**
     * 将语言代码转换为可视化显示名称
     * 用于将loadSourceLanguages和loadTargetLanguages中获取到的language转化为用户友好的显示名称
     *
     * @param language 语言代码
     * @return 语言显示名称
     */
    @NonNull
    String getLanguageDisplayName(String language);

    /**
     * 翻译开始前的准备工作，运行在UI线程
     * 在onStart前调用
     */
    void beforeStart();

    /**
     * 翻译开始前的准备工作，运行在子线程
     */
    void onStart();

    /**
     * 执行文本翻译
     *
     * @param text           待翻译文本，其值受Configuration.acceptTranslated影响
     * @param sourceLanguage 源语言
     * @param targetLanguage 目标语言
     * @return 翻译结果
     * @throws IOException 翻译过程中的IO异常
     */
    @NonNull
    String translate(String text, String sourceLanguage, String targetLanguage) throws IOException;

    /**
     * 翻译结束或用户取消后的清理工作，运行在子线程
     */
    void onFinish();

    /**
     * 翻译结束或用户取消后的清理工作，运行在UI线程
     * 在onFinish后调用
     */
    void afterFinish();

    /**
     * 翻译错误处理回调，运行在UI线程
     *
     * @param e 异常信息
     * @return 返回true时MT不会显示错误信息对话框
     */
    boolean onError(Exception e);

    /**
     * 翻译引擎配置构建器
     */
    class ConfigurationBuilder {
        private boolean acceptTranslated;
        private boolean forceNotToSkipTranslated;
        private boolean targetLanguageMutable;
        private boolean autoRepairFormatControlError;
        private boolean disableAutoHideLanguage;

        /**
         * 设置是否接受翻译过的文本。例如一个词条"Apple"，第一次翻译将传递"Apple"，
         * 假设翻译结果为"苹果"，如果再次对这个词条进行翻译，
         * 当acceptTranslated为false，将传递"Apple"进行翻译；
         * 当acceptTranslated为true，将传递"苹果"进行翻译。
         * <p>
         * 默认为false
         */
        public ConfigurationBuilder setAcceptTranslated(boolean acceptTranslated) {
            this.acceptTranslated = acceptTranslated;
            return this;
        }

        /**
         * 设置是否强制不跳过已翻译词条（开启后界面上将会隐藏该选项）
         * <p>
         * 默认为false
         */
        public ConfigurationBuilder setForceNotToSkipTranslated(boolean forceNotToSkipTranslated) {
            this.forceNotToSkipTranslated = forceNotToSkipTranslated;
            return this;
        }

        /**
         * 设置TargetLanguage是否是可变的，如果翻译引擎的目标语言列表会随着
         * SourceLanguage的变化而变化，值应该设为true
         * <p>
         * 默认为false
         */
        public ConfigurationBuilder setTargetLanguageMutable(boolean targetLanguageMutable) {
            this.targetLanguageMutable = targetLanguageMutable;
            return this;
        }

        /**
         * 设置是否自动修复简单的控制符翻译错误，如%s被翻译为％S
         * <p>
         * 默认为false
         */
        public ConfigurationBuilder setAutoRepairFormatControlError(boolean autoRepairFormatControlError) {
            this.autoRepairFormatControlError = autoRepairFormatControlError;
            return this;
        }

        /**
         * 设置是否自动隐藏语言
         * 像谷歌翻译这类翻译引擎支持大量语言翻译，
         * 用户可在翻译模式中选择要隐藏的语言，如果不希望翻译引擎的源语言和
         * 目标语言被自动隐藏，可将值设为true
         * <p>
         * 默认为false
         */
        public ConfigurationBuilder setDisableAutoHideLanguage(boolean disableAutoHideLanguage) {
            this.disableAutoHideLanguage = disableAutoHideLanguage;
            return this;
        }

        /**
         * 构建配置对象
         *
         * @return Configuration实例
         */
        public Configuration build() {
            return new Configuration(acceptTranslated,
                    forceNotToSkipTranslated,
                    targetLanguageMutable,
                    autoRepairFormatControlError,
                    disableAutoHideLanguage);
        }
    }

    /**
     * 翻译引擎配置类
     * 包含翻译引擎的各项配置参数
     */
    class Configuration {
        /**
         * 是否接受已翻译的文本
         */
        public final boolean acceptTranslated;
        /**
         * 是否强制不跳过已翻译词条
         */
        public final boolean forceNotToSkipTranslated;
        /**
         * 目标语言是否可变
         */
        public final boolean targetLanguageMutable;
        /**
         * 是否自动修复格式控制符错误
         */
        public final boolean autoRepairFormatControlError;
        /**
         * 是否禁用自动隐藏语言
         */
        public final boolean disableAutoHideLanguage;

        /**
         * 构造配置对象
         */
        Configuration(boolean acceptTranslated,
                      boolean forceNotToSkipTranslated,
                      boolean targetLanguageMutable,
                      boolean autoRepairFormatControlError,
                      boolean disableAutoHideLanguage) {
            this.acceptTranslated = acceptTranslated;
            this.forceNotToSkipTranslated = forceNotToSkipTranslated;
            this.targetLanguageMutable = targetLanguageMutable;
            this.autoRepairFormatControlError = autoRepairFormatControlError;
            this.disableAutoHideLanguage = disableAutoHideLanguage;
        }
    }
}
